# -*- coding: utf-8 -*-

# Deploy pipeline: https://tsum.yandex-team.ru/pipe/projects/report/delivery-dashboard/ShinySpokTemplate

import json
import enum
import logging
import re
from sandbox import sdk2
from sandbox.sdk2.vcs.svn import Arcadia
from sandbox.sdk2.helpers import process
import os
from sandbox.projects.common.arcadia import sdk
from sandbox.projects.common.vcs import arc
import sandbox.projects.common.constants as consts
from sandbox.projects.common.binary_task import deprecated as binary_task
from sandbox.projects.market.infra.helpers import arcadia as arcadia_helpers, app_producer

SHINY = 'shiny'
USERVER = 'userver'
BRANCH_TEMPLATE = 'users/{arcadia_user}/spok_{template_name}_{app_name}'

def update_review(mount_point, branch_name, commit_message, review_message):
    logging.info('Update Review for {}'.format(branch_name))

    arc_status = arc.Arc().status(mount_point=mount_point, as_dict=True)
    logging.debug('Arc status: {}'.format(arc_status))

    arc.Arc().checkout(mount_point, branch=branch_name, create_branch=True)
    arc.Arc().add(mount_point, all_changes=True)
    arc.Arc().commit(mount_point, message=commit_message, all_changes=True)
    arc.Arc().push(mount_point, upstream=branch_name, force=True)

    try:
        pr_info = arc.Arc().pr_create(
            mount_point=mount_point,
            message=review_message,
            publish=True,
            auto=True,
        ).decode()
    except arc.ArcCommandFailed:
        try:
            status = arc.pr_status(mount_point=mount_point).decode()
            pr_info = 'Review updated:\n{}'.format(status)
        except arc.ArcCommandFailed:
            pr_info = 'Unable to create or update review for {}'.format(branch_name)

    logging.info(pr_info)

    return pr_info


class AppInfo:
    def __init__(self, params, mount_point):
        from market.library.shiny.server.gen.lib.gen_app import make_kebab_name

        self.dest_app_dir = params.dest_dir.strip('/')
        self.application_name = params.application_name
        self.kebab_app_name = make_kebab_name(self.application_name)
        self.userver_kebab_app_name = 'market-' + self.kebab_app_name
        self.dest_app_dir_abs = os.path.join(mount_point, self.dest_app_dir)
        self.app_binary_name = str(params.bin_name or self.kebab_app_name)
        self.rtc_json_path = os.path.join(self.dest_app_dir, 'deploy', 'rtc.json')
    
    def validate_for_userver(self):
        dest_app_dir_pattern = 'taxi/uservices/services/' + self.userver_kebab_app_name
        if dest_app_dir_pattern != self.dest_app_dir:
            raise RuntimeError('{} != {}'.format(dest_app_dir_pattern, self.dest_app_dir))

        app_folder_name = self.dest_app_dir.split('/')[-1]
        if app_folder_name != self.userver_kebab_app_name:
            raise RuntimeError('{} != {}'.format(app_folder_name, self.userver_kebab_app_name))

# FIXME should be based on MarketApplicationArcProducerBase but without instant PR merge
class MarketCppApplicationProducer(binary_task.LastBinaryTaskRelease, sdk2.Task):
    class Parameters(sdk2.Task.Parameters):
        dest_dir = sdk2.parameters.String("Destination dir in VCS", required=True)
        application_name = sdk2.parameters.String("Application name (ex: market-team-application)", required=True)
        bin_name = sdk2.parameters.String("C++ binary name")
        ya_owner = sdk2.parameters.String("Owner for ya.make (ex: g:marketinfra)", required=True)
        commit_message = sdk2.parameters.String("Commit message for SVN", required=True)
        arcadia_user = sdk2.parameters.String('Arcadia user', default='robot-market-infra')
        description = sdk2.parameters.String('App description', default='Spok C++ app')
        ext_params = binary_task.binary_release_parameters(stable=True)

        with sdk2.parameters.RadioGroup("Template Name") as template_name:
            template_name.values[USERVER] = template_name.Value("Taxi uServer")
            template_name.values[SHINY] = template_name.Value("Market Shiny", default=True)
        
        with template_name.value[SHINY]:
            cpp_namespace = sdk2.parameters.String("C++ namespace (ex: NMyApp)", required=True)
        
        with sdk2.parameters.Output:
            arcadia_url = sdk2.parameters.String("Arcadia URL", default="")


    class Requirements(sdk2.Requirements):
        # https://st.yandex-team.ru/MARKETDX-1020#62446b4674b4f51616976da5
        container_resource = 2935522420

    def __generate_shiny_app(self, mount_point, app_info):
        from market.library.shiny.server.gen.lib.gen_app import generate_app, make_kebab_name

        # Generate app template here
        generate_app(
            app_info.application_name,
            app_info.dest_app_dir_abs,
            self.Parameters.cpp_namespace,
            app_info.app_binary_name,
            self.Parameters.ya_owner,
            self.Parameters.description,
        )

        app_producer.process_parent_ya_make(app_info.dest_app_dir_abs, str(self.Parameters.ya_owner))


    def __generate_userver_app(self, mount_point, app_info):
        from ruamel.yaml import YAML, representer

        class NonAliasingRTRepresenter(representer.RoundTripRepresenter):
            def ignore_aliases(self, _):
                return True

        app_info.validate_for_userver()

        with sdk2.helpers.ProcessLog(self, logger="uservices") as pl:
            sdk2.helpers.subprocess.check_call(
                [arc.Arc().binary_path, "config", "user.full_name", "Miley Cyrus"],
                cwd=os.path.join(mount_point, 'taxi/uservices'),
                stdout=pl.stdout,
                stderr=pl.stderr,
            )

            sdk2.helpers.subprocess.check_call(
                [arc.Arc().binary_path, "config", "user.email", "{}@yandex-team.ru".format(self.Parameters.arcadia_user)],
                cwd=os.path.join(mount_point, 'taxi/uservices'),
                stdout=pl.stdout,
                stderr=pl.stderr,
            )

            sdk2.helpers.subprocess.check_call(
                ["bash", "./scripts/create-new-service.sh", app_info.userver_kebab_app_name, "market"],
                cwd=os.path.join(mount_point, 'taxi/uservices'),
                stdout=pl.stdout,
                stderr=pl.stderr,
            )

            with open(os.path.join(app_info.dest_app_dir_abs, "service.yaml"), "r+") as f:
                yaml = YAML()
                yaml.Representer = NonAliasingRTRepresenter
                service_yaml = yaml.load(f)

                ya_make_owners = [str(name) for name in self.Parameters.ya_owner.split(' ')]

                service_yaml["wiki"] = "https://wiki.yandex-team.ru/market"
                service_yaml["maintainers"] = ya_make_owners 
                service_yaml.pop("clownductor_service_info", None)
                service_yaml.pop("dashboards", None)
                service_yaml["uservice_unit"]["description"] = str(self.Parameters.description)

                service_yaml["ya-make"] = {
                    "enabled": True,
                    "owners": ya_make_owners,
                }

                service_yaml["market-spok-infra"] = {
                    "enabled": True,
                    "description": str(self.Parameters.description),
                    "bin_name": app_info.app_binary_name,
                }

                f.seek(0)
                f.truncate()

                yaml.dump(service_yaml, f) 

            sdk2.helpers.subprocess.check_call(
                ["make", "gen-{}".format(app_info.userver_kebab_app_name)],
                cwd=os.path.join(mount_point, 'taxi/uservices'),
                stdout=pl.stdout,
                stderr=pl.stderr,
            )

            sdk2.helpers.subprocess.check_call(
                ["make", "arc-gen-{}".format(app_info.userver_kebab_app_name)],
                cwd=os.path.join(mount_point, 'taxi/uservices'),
                stdout=pl.stdout,
                stderr=pl.stderr,
            )

            sdk2.helpers.subprocess.check_call(
                ["taxi-format", os.path.join("./services", app_info.userver_kebab_app_name)],
                cwd=os.path.join(mount_point, 'taxi/uservices'),
                stdout=pl.stdout,
                stderr=pl.stderr,
            )

    def __build_app(self, mount_point, app_info):
        # Run build after all the new files have been added, cause the build will create some garbage
        logging.info("Running build")
        sdk.do_build(consts.YMAKE_BUILD_SYSTEM, mount_point, [app_info.dest_app_dir], test=True, clear_build=False)
        logging.info("Build finished")

        logging.info("Running package build")
        sdk.do_package(mount_point, [app_info.rtc_json_path], clear_build=False, direct_arcadia_mode=True)
        logging.info("Package build finished")

    def __set_review_info(self, review_info):
        review = re.findall(r'https://a\.yandex-team\.ru/review/(\d+)', review_info)
        if not review:
           raise RuntimeError('Review url not found in {}'.format(review_info)) 

        self.Parameters.arcadia_url = "https://a.yandex-team.ru/review/{}".format(review[0])

    def __generate_app(self, template_name):
        logging.info('Start app generation')
        with arc.Arc().mount_path(None, "trunk", fetch_all=False) as mount_point:
            logging.info('Arcadia mounted')

            app_info = AppInfo(self.Parameters, mount_point)

            if os.path.exists(app_info.dest_app_dir_abs):
                package_json_path = os.path.join(app_info.dest_app_dir_abs, 'deploy/rtc.json')
                if app_producer.is_package_json_for_current_app(app_info.application_name, package_json_path):
                    logging.info('Code for this App already generated. Do nothing')
                    self.Parameters.arcadia_url = 'Nothing to commit'
                    return
                else:
                    raise RuntimeError('Path "{}" already exists'.format(app_info.dest_app_dir_abs))

            if template_name == SHINY:
                self.__generate_shiny_app(mount_point, app_info)    
            elif template_name == USERVER:
                self.__generate_userver_app(mount_point, app_info)    
            else:
                raise RuntimeError('Unknown template_name: {}'.format(self.Parameters.template_name))

            self.__build_app(mount_point, app_info)

            branch_name = BRANCH_TEMPLATE.format(
                arcadia_user=self.Parameters.arcadia_user,
                template_name=template_name,
                app_name=app_info.kebab_app_name
            )

            review_info = update_review(
                mount_point=mount_point,
                branch_name=branch_name,
                commit_message=str(self.Parameters.commit_message),
                review_message=str(self.Parameters.commit_message),
            )

            self.__set_review_info(review_info)

    def on_execute(self):
        super(MarketCppApplicationProducer, self).on_execute()

        if self.Requirements.tasks_resource is None:
            return

        logging.info('Selected template_name: {}'.format(self.Parameters.template_name))
        self.__generate_app(self.Parameters.template_name)
