# coding=utf-8

import calendar
import json
import logging
from contextlib import contextmanager

from sandbox.common import errors
from sandbox.projects.common.arcadia import sdk
from sandbox.projects.market.infra import TsumJsonResource
from sandbox.sdk2 import Task, parameters, ResourceData
from sandbox.sdk2 import svn

logger = logging.getLogger(__name__)

DEFAULT_TO_REVISION = 'HEAD'


class MarketGenerateChangelogTask(Task):
    """Deprecated; use MARKET_GET_ARCADIA_DEPENDENCIES and GenerateArcadiaChangelogJob in TSUM instead
       Generates true changelog for Market arcadia modules
    """

    class Parameters(Task.Parameters):
        kill_timeout = 900

        module_path = parameters.String(
            'Module path',
            description='Folder with ya.make (Example: market/backctld)',
            required=True
        )
        from_revision = parameters.String(
            'From revision (exclusive)',
            required=True,
        )
        to_revision = parameters.String(
            'To revision',
            default_value=DEFAULT_TO_REVISION
        )
        current_branch = parameters.ArcadiaUrl(
            'Branch path',
            description='Arcadia URL',
        )

        with parameters.Output:
            tsum_json_resource = parameters.Resource('package.json', resource_type=TsumJsonResource)

    def _make_tsum_json_resource(self, changes):
        self.Parameters.tsum_json_resource = TsumJsonResource(self, description='packages.json', path='packages.json')

        app_res = ResourceData(self.Parameters.tsum_json_resource)
        app_res.path.write_bytes(json.dumps({'changelog': changes}))
        app_res.ready()

    @staticmethod
    def _filter_changes(svn_log, allowed_paths):
        changes = []
        for log_entry in svn_log:
            # Get paths from dicts and remove the branch substring
            changed_paths = [''.join(path_dict['text'].split('arcadia/', 1)[1:]) for path_dict in log_entry['paths']]
            for changed_path in changed_paths:
                is_found_match = False
                for allowed_path in allowed_paths:
                    # It should be faster then startswith but less accurate because substring `changed_path` could be
                    # in the middle or in the end of `allowed_path`
                    if allowed_path in changed_path:
                        # Ensure `allowed_path` is the prefix of `changed_path`
                        if changed_path.startswith(allowed_path):
                            logger.info('Commit r%s has been added into the list of changes', log_entry['revision'])
                            changes.append({
                                'author': log_entry['author'],
                                'revision': log_entry['revision'],
                                'change': log_entry['msg'],
                                'timestampSeconds': calendar.timegm(log_entry['date'].utctimetuple())
                            })
                            # Stop processing this `log_entry`
                            is_found_match = True
                            break
                if is_found_match:
                    break

        logger.info('The number of commits is %s', len(changes))
        return changes

    @contextmanager
    def get_arcadia(self, arcadia_url):
        try:
            with sdk.mount_arc_path(arcadia_url, use_arc_instead_of_aapi=True) as arcadia:
                yield arcadia
        except errors.TaskFailure as e:
            logger.exception(e)
            yield svn.Arcadia.get_arcadia_src_dir(arcadia_url)

    def on_execute(self):
        arcadia_url = self.Parameters.current_branch

        if len(arcadia_url.rsplit('/arcadia', 1)) == 1 and len(arcadia_url.rsplit('@', 1)) == 1:
            # Use only the repository Arcadia
            arcadia_url += '/arcadia'

        from_revision = str(self.Parameters.from_revision).strip()
        to_revision = str(self.Parameters.to_revision).strip() or DEFAULT_TO_REVISION

        if to_revision != 'HEAD':
            arcadia_url += '@{}'.format(to_revision)
        logger.info('Arcadia URL is %s', arcadia_url)

        logger.info('Getting commits from %s to %s', from_revision, to_revision)
        svn_log = svn.Arcadia.log(
            url=arcadia_url,
            revision_from=from_revision,
            revision_to=to_revision,
            fullpath=True,
            timeout=600
        )
        if svn_log and str(svn_log[0]['revision']) == from_revision:
            svn_log = svn_log[1:]
        logger.info('Found %s commits', len(svn_log))

        if not sdk.fuse_available():
            raise Exception('Fuse unavailable')

        module_path = str(self.Parameters.module_path)
        with self.get_arcadia(arcadia_url) as arcadia:
            logging.info('Getting %s dependencies', self.Parameters.module_path)
            dumped_deps = sdk.dump_targets_deps(source_root=arcadia, targets=[module_path], plain_mode=True)
            logging.info('Found %s dependencies' % len(dumped_deps))

        # The list of paths to filter changed files and directories.
        # Should ends with "/" to prevent match in the middle of a word,
        # e.g. without the slash "contrib/python/py" matches "contrib/python/pytest...".
        allowed_paths = [dep + '/' for dep in dumped_deps]
        module_path += '/'

        if any(module_path == p for p in allowed_paths):
            logging.info('allowed_paths already contains module_path')
        else:
            logging.info('allowed_paths doesn\'t contain module_path. Adding to ...')
            allowed_paths.append(module_path)

        logger.info('The number of paths from dependencies is %s', len(allowed_paths))

        changes = self._filter_changes(svn_log=svn_log, allowed_paths=allowed_paths)
        self._make_tsum_json_resource(changes=changes)
