# coding=utf-8

import os
import json
import logging
from contextlib import contextmanager

from sandbox.common import errors
from sandbox.projects.common.arcadia import sdk
from sandbox.projects.market.infra import TsumJsonResource
from sandbox.sdk2 import Task, parameters, ResourceData
from sandbox.sdk2 import svn

logger = logging.getLogger(__name__)


class MarketGetArcadiaDependenciesTask(Task):
    """Generates dependency list for Market arcadia modules
    """

    class Parameters(Task.Parameters):
        kill_timeout = 900

        module_path = parameters.String(
            'Module path',
            description='Folder with ya.make (Example: market/backctld)',
            required=True
        )
        current_branch = parameters.ArcadiaUrl(
            'Branch path',
            description='Arcadia URL',
        )

        with parameters.Output:
            tsum_json_resource = parameters.Resource('package.json', resource_type=TsumJsonResource)

    def _make_tsum_json_resource(self, paths):
        self.Parameters.tsum_json_resource = TsumJsonResource(self, description='packages.json', path='packages.json')

        app_res = ResourceData(self.Parameters.tsum_json_resource)
        app_res.path.write_bytes(json.dumps({'paths': paths}))
        app_res.ready()

    @contextmanager
    def get_arcadia(self, arcadia_url):
        try:
            with sdk.mount_arc_path(arcadia_url, use_arc_instead_of_aapi=True) as arcadia:
                yield arcadia
        except errors.TaskFailure as e:
            logger.exception(e)
            yield svn.Arcadia.get_arcadia_src_dir(arcadia_url)

    def on_execute(self):
        arcadia_url = self.Parameters.current_branch

        if len(arcadia_url.rsplit('/arcadia', 1)) == 1 and len(arcadia_url.rsplit('@', 1)) == 1:
            # Use only the repository Arcadia
            arcadia_url += '/arcadia'

        logger.info('Arcadia URL is %s', arcadia_url)

        if not sdk.fuse_available():
            raise Exception('Fuse unavailable')

        with self.get_arcadia(arcadia_url) as arcadia:
            targets = []
            logging.info('Getting %s dependencies', self.Parameters.module_path)
            if str(self.Parameters.module_path).endswith('.json'):
                abs_target = os.path.join(arcadia, str(self.Parameters.module_path))
                with open(abs_target) as f:
                    package_info = json.load(f)
                    if 'build' in package_info and 'targets' in package_info['build']:
                        targets = package_info['build']['targets']
            else:
                targets = str(self.Parameters.module_path).split(',')
            dumped_deps = sdk.dump_targets_deps(source_root=arcadia, targets=targets, plain_mode=True)
            logging.info('Found %s dependencies' % len(dumped_deps))

        # The list of paths to filter changed files and directories.
        # Should ends with "/" to prevent match in the middle of a word,
        # e.g. without the slash "contrib/python/py" matches "contrib/python/pytest...".
        allowed_paths = [dep + '/' for dep in dumped_deps]
        logger.info('The number of paths from dependencies is %s', len(allowed_paths))

        self._make_tsum_json_resource(paths=allowed_paths)
