# coding=utf-8
import logging
import os
import io
import jinja2
import sys
from shutil import copytree, ignore_patterns
from sandbox import common
from sandbox import sdk2
from sandbox.sdk2.helpers import subprocess as sp, ProcessLog
from sandbox.projects.common.arcadia import sdk
import sandbox.projects.common.constants as consts
import sandbox.projects.market.infra.helpers.app_producer as producer
from sandbox.projects.market.infra.MarketApplicationArcProducerBase import MarketApplicationArcProducerBase

REVIEWS_URL = 'https://a.yandex-team.ru/review/'
WAIT_TIME = 10


class MarketGolangApplicationArcProducer(MarketApplicationArcProducerBase):
    arc_token = None

    class Parameters(MarketApplicationArcProducerBase.Parameters):
        source_dir = sdk2.parameters.String("Template go application dir in Arcadia",
                                            default='/market/devexp/app_templates/go_app/', required=False)
        dest_dir = sdk2.parameters.String("Destination dir in Arcadia", required=True)
        ya_owner = sdk2.parameters.String("Owner for ya.make (ex: g:marketinfra)", required=True)
        maintainer = sdk2.parameters.String("Spok author / maintainer of service", required=True)
        ya_package_config_path = sdk2.parameters.String("Path to ya package config", required=False)

    @property
    def base_branch_name(self):
        return 'MarketGolangApplicationArcProducer'

    def produce_application(self):
        logging.info("Starting MarketGolangApplicationArcProducer task with params: %s", self.Parameters)

        source_app_dir = self.Parameters.source_dir.strip('/')
        dest_app_dir = self.Parameters.dest_dir.strip('/')
        template_dir = os.path.join(self.arcadia_root, source_app_dir)
        dest_dir = os.path.join(self.arcadia_root, dest_app_dir)

        if self.Parameters.ya_package_config_path:
            package_json_path = os.path.join(self.arcadia_root, self.Parameters.ya_package_config_path)
        else:
            package_json_path = os.path.join(dest_dir, 'package.json')
        if os.path.exists(dest_dir):
            if producer.is_package_json_for_current_app(self.Parameters.application_name, package_json_path):
                logging.info('Code for this App already generated. Do nothing')
                return False
            else:
                raise RuntimeError('Path "{}" already exists'.format(dest_dir))

        self.prepare_app(self.arcadia_root, template_dir, dest_dir, self.Parameters)
        producer.process_parent_ya_make(dest_dir, self.Parameters.ya_owner)

        logging.info("Running build")
        sdk.do_build(
            consts.YMAKE_BUILD_SYSTEM, self.arcadia_root, [dest_app_dir], test=True, checkout=True, clear_build=False
        )

        sdk.do_package(self.arcadia_root, ["%s/package.json" % dest_app_dir],
            clear_build=False, checkout=True, convert=True, direct_arcadia_mode=True)
        logging.info('Application created!')

        return True

    def python(self):
        if not common.system.inside_the_binary():
            return sys.executable
        else:
            return '/skynet/python/bin/python'

    def get_ya_tool(self, arcadia_root, frepkage_root=None):
        if frepkage_root:
            return os.path.join(frepkage_root, 'ya-bin')

        ya_path = os.path.join(arcadia_root, 'ya')
        if os.path.exists(ya_path):
            return ya_path
        return os.path.join(arcadia_root, 'devtools', 'ya', 'ya')

    def prepare_app(self, arcadia_root, copy_from, copy_to, params):
        copytree(copy_from, copy_to, ignore=ignore_patterns(".svn"))

        for root, folders, files in os.walk(copy_to):
            for file_name in files:
                abs_filepath = os.path.join(root, file_name)

                filename_without_ext, file_extension = os.path.splitext(file_name)
                new_path = os.path.join(root, filename_without_ext)

                if file_extension == '.jinja2':
                    logging.info("Processing jinja2 template file: %s", abs_filepath)
                    logging.info("Copying template to: %s", new_path)

                    with io.open(abs_filepath, "r", encoding="utf-8") as file:
                        tmpl = file.read()

                    with io.open(new_path, "w", encoding="utf-8") as file:
                        template_context = {
                            "app_name": params.application_name,
                            "app_path": self.Parameters.dest_dir.strip('/'),
                            "maintainer": params.maintainer,
                            "ya_owner": params.ya_owner
                        }

                        tmpl_processed = jinja2.Template(tmpl, keep_trailing_newline=True).render(template_context)
                        file.write(tmpl_processed)

                    logging.info("Removing old template: %s", abs_filepath)
                    os.remove(abs_filepath)

                if root == copy_to and filename_without_ext == "package.json" and params.ya_package_config_path:
                    ya_package_config_name = os.path.basename(params.ya_package_config_path)
                    ya_package_config_dir = os.path.join(arcadia_root, os.path.dirname(params.ya_package_config_path))
                    if copy_to != ya_package_config_dir or ya_package_config_name != "package.json":
                        os.replace(os.path.join(root, "package.json"), os.path.join(arcadia_root,
                                                                                    params.ya_package_config_path))

                # Может случиться так, что из-за названия проекта, нарушится лексикографический порядок в импортах.
                # Это сломает style-тесты. После обработки шаблона ya style должен пофиксить это в go файлах.
                filename_without_ext, file_extension = os.path.splitext(filename_without_ext)
                if file_extension == '.go':
                    logging.info("Fix style for go file: %s", new_path)
                    with ProcessLog(self, logger="ya_style_log") as pl:
                        environment = os.environ.copy()
                        ya_style_cmd = [
                            self.python(),
                            self.get_ya_tool(arcadia_root),
                            'style',
                            new_path
                        ]
                        sp.call(
                            ' '.join(ya_style_cmd),
                            shell=True,
                            env=environment,
                            cwd=root,
                            stdout=pl.stdout,
                            stderr=sp.STDOUT
                        )
