# coding=utf-8
import logging
import os
import io
import jinja2
import sys
from shutil import copytree, ignore_patterns
from sandbox import common
from sandbox import sdk2
from sandbox.sdk2.vcs.svn import Arcadia
from sandbox.sdk2.helpers import subprocess as sp, ProcessLog
from sandbox.projects.common.arcadia import sdk
import sandbox.projects.common.constants as consts
import sandbox.projects.market.infra.helpers.arcadia as arcadia_helpers
import sandbox.projects.market.infra.helpers.app_producer as producer


class MarketGolangApplicationProducer(sdk2.Task):
    class Parameters(sdk2.Task.Parameters):
        source_dir = sdk2.parameters.String("Template go application dir in Arcadia",
                                            default='/market/infra/go_app/', required=False)
        dest_dir = sdk2.parameters.String("Destination dir in Arcadia", required=True)
        application_name = sdk2.parameters.String("Application name (ex: market-team-application)", required=True)
        ya_owner = sdk2.parameters.String("Owner for ya.make (ex: g:marketinfra)", required=True)
        commit_message = sdk2.parameters.String("Commit message for SVN", required=True)
        maintainer = sdk2.parameters.String("Spok author / maintainer of service", required=True)
        arcadia_user = sdk2.parameters.String('Arcadia user', default='zomb-sandbox-rw')
        ya_package_config_path = sdk2.parameters.String("Path to ya package config", required=False)

    def on_execute(self):
        logging.info("Starting MarketGoApplicationProducer task with params: %s", self.Parameters)

        template_dir = Arcadia.checkout(
            'arcadia:/arc/trunk/arcadia/' + self.Parameters.source_dir, path=str(self.path("template"))
        )

        dest_app_dir = self.Parameters.dest_dir.strip('/')

        arcadia_dir = arcadia_helpers.checkout_immediates(str(self.path("arcadia")), dest_app_dir)

        dest_dir = os.path.join(arcadia_dir, dest_app_dir)
        if self.Parameters.ya_package_config_path:
            package_json_path = os.path.join(self.arcadia_root, self.Parameters.ya_package_config_path)
        else:
            package_json_path = os.path.join(dest_dir, 'package.json')
        if os.path.exists(dest_dir):
            if producer.is_package_json_for_current_app(self.Parameters.application_name, package_json_path):
                logging.info('Code for this App already generated. Do nothing')
                return
            else:
                raise RuntimeError('Path "{}" already exists'.format(dest_dir))

        self.prepare_app(arcadia_dir, template_dir, dest_dir, self.Parameters)

        arcadia_helpers.safe_add_files(arcadia_dir)

        logging.info("Running build")
        sdk.do_build(
            consts.YMAKE_BUILD_SYSTEM, arcadia_dir, [dest_app_dir], test=True, checkout=True, clear_build=False
        )

        ya_package_source_root = os.path.dirname(arcadia_dir)

        sdk.do_package(ya_package_source_root, ["%s/package.json" % dest_app_dir], clear_build=False, checkout=True, convert=True)

        logging.info('Commiting to arcadia with message: %s', self.Parameters.commit_message)
        Arcadia.commit(arcadia_dir, self.Parameters.commit_message, self.Parameters.arcadia_user)
        logging.info('Application created!')

    def python(self):
        if not common.system.inside_the_binary():
            return sys.executable
        else:
            return '/skynet/python/bin/python'

    def get_ya_tool(self, arcadia_root, frepkage_root=None):
        if frepkage_root:
            return os.path.join(frepkage_root, 'ya-bin')

        ya_path = os.path.join(arcadia_root, 'ya')
        if os.path.exists(ya_path):
            return ya_path
        return os.path.join(arcadia_root, 'devtools', 'ya', 'ya')

    def prepare_app(self, arcadia_root, copy_from, copy_to, params):
        copytree(copy_from, copy_to, ignore=ignore_patterns(".svn"))

        for root, folders, files in os.walk(copy_to):
            for file_name in files:
                abs_filepath = os.path.join(root, file_name)

                filename_without_ext, file_extension = os.path.splitext(file_name)
                new_path = os.path.join(root, filename_without_ext)

                if file_extension == '.jinja2':
                    logging.info("Processing jinja2 template file: %s", abs_filepath)
                    logging.info("Copying template to: %s", new_path)

                    with io.open(abs_filepath, "r", encoding="utf-8") as file:
                        tmpl = file.read()

                    with io.open(new_path, "w", encoding="utf-8") as file:
                        template_context = {
                            "app_name": params.application_name,
                            "app_path": self.Parameters.dest_dir.strip('/'),
                            "maintainer": params.maintainer,
                            "ya_owner": params.ya_owner
                        }

                        tmpl_processed = jinja2.Template(tmpl, keep_trailing_newline=True).render(template_context)
                        file.write(tmpl_processed)

                    logging.info("Removing old template: %s", abs_filepath)
                    os.remove(abs_filepath)

                if root == copy_to and filename_without_ext == "package.json" and params.ya_package_config_path:
                    ya_package_config_name = os.path.basename(params.ya_package_config_path)
                    ya_package_config_dir = os.path.join(arcadia_root,
                                                         os.path.dirname(params.ya_package_config_path))
                    if copy_to != ya_package_config_dir or ya_package_config_name != "package.json":
                        os.replace(os.path.join(root, "package.json"), os.path.join(arcadia_root,
                                                                                    params.ya_package_config_path))
                # Может случиться так, что из-за названия проекта, нарушится лексикографический порядок в импортах.
                # Это сломает style-тесты. После обработки шаблона ya style должен пофиксить это в go файлах.
                filename_without_ext, file_extension = os.path.splitext(filename_without_ext)
                if file_extension == '.go':
                    logging.info("Fix style for go file: %s", new_path)
                    with ProcessLog(self, logger="ya_style_log") as pl:
                        environment = os.environ.copy()
                        ya_style_cmd = [
                            self.python(),
                            self.get_ya_tool(arcadia_root),
                            'style',
                            new_path
                        ]
                        sp.call(
                            ' '.join(ya_style_cmd),
                            shell=True,
                            env=environment,
                            cwd=root,
                            stdout=pl.stdout,
                            stderr=sp.STDOUT
                        )
