# coding=utf-8
import json
import logging
import os
import time

import requests
from sandbox import sdk2, common

from sandbox.common.types.client import Tag
from sandbox.common.types.misc import DnsType
from sandbox.projects.common import file_utils
from sandbox.sdk2.helpers import subprocess
from sandbox.sdk2.vcs.git import Git

logger = logging.getLogger(__name__)


class MarketHealthIntegrationTests(sdk2.Task):
    """Runs integration tests for market-health
    """

    class Requirements(sdk2.Requirements):
        privileged = True
        dns = DnsType.DNS64
        client_tags = Tag.IPV6

    class Parameters(sdk2.Task.Parameters):
        container = sdk2.parameters.Container(
            "Образ LXC-Контейнера с Docker'ом и Docker Compose'ом",
            default_value=1046359220,
            required=True,
        )

        commit_sha1 = sdk2.parameters.String(
            "SHA1 коммита, на котором нужно запустить тесты",
            default="master",
            required=True
        )

    def on_prepare(self):
        # В on_prepare, а не on_execute потому что клонирование пишет в кеш Git-репозиториев, и это нельзя делать в
        # on_execute в тасках с privileged = True.
        self._clone_market_health_repository()

    def on_execute(self):
        self._collect_debug_info_about_mount_points()
        self._configure_and_launch_docker_daemon()
        self._launch_docker_containers()
        self._assemble_project()
        self._wait_for_start_clickhouse()
        self._run_integration_tests()

    def _clone_market_health_repository(self):
        git = Git('https://github.yandex-team.ru/market-infra/market-health.git')
        git.clone(self._market_health_repository_dir, "master")
        git.update_cache_repo("refs/pull/*")
        git.execute("checkout", "-f", self.Parameters.commit_sha1, cwd=self._market_health_repository_dir)

    def _configure_and_launch_docker_daemon(self):
        self._fix_ipv6()
        self._write_docker_config()
        self._restart_docker()

    def _fix_ipv6(self):
        self._collect_debug_info_about_network()
        try:
            # Костыли для IPv6 в Докер-контейнерах, взяты из скрипта таски BuildDockerImageV6
            # https://st.yandex-team.ru/MARKETINFRA-3566#5d307243a2b79e001c014ea3
            self._run(['ip6tables', '-t', 'nat', '-A', 'POSTROUTING', '-s', 'fd00::/8', '-j', 'MASQUERADE'])
            self._run(['sysctl', 'net.ipv6.conf.default.accept_ra=2'])
            self._run(['sysctl', 'net.ipv6.conf.all.forwarding=1'])
        finally:
            self._collect_debug_info_about_network()

    def _write_docker_config(self):
        self._write_json_to_log_and_file(
            '/etc/docker/daemon.json',
            {
                # Рекомендован здесь: https://docs.docker.com/storage/storagedriver/select-storage-driver/
                'storage-driver': 'overlay2',
                # Данные Докера должны быть в рабочей папке таски потому что вне рабочей папки OverlayFS.
                # Драйвер overlay2 не работает на OverlayFS.
                'data-root': self._docker_data_dir,
                # Включаем IPv6
                "ipv6": True,
                # Подсеть, из которой Докер будет выдавать IPv6-адреса контейнеров
                "fixed-cidr-v6": "fd00::/8",
                # DNS-серверы, которые умеют отдавать IPv6-адреса для IPv4-only хостов
                "dns": ["2a02:6b8:0:3400::1023", "2a02:6b8:0:3400::5005"],
                # Говорим Докеру не настраивать сеть самостоятельно, у него не получается
                "iptables": False,
                "ip-forward": False,
            }
        )

    def _restart_docker(self):
        try:
            self._run(['systemctl', 'restart', 'docker.service'])
        finally:
            self._collect_debug_info_about_docker()

    def _launch_docker_containers(self):
        try:
            self._run_with_custom_logger(
                'docker',
                [
                    'docker-compose',
                    '-f',
                    os.path.join(
                        self._market_health_repository_dir,
                        'health-integration-tests',
                        'docker',
                        'docker-compose.yml'
                    ),
                    'up',
                    '-d'
                ]
            )
        except sdk2.helpers.ProcessLog.CalledProcessError as e:
            raise self.IntegrationTestsError(e)

    def _assemble_project(self):
        try:
            self._run_with_custom_logger(
                'tests',
                [
                    os.path.join(self._market_health_repository_dir, 'gradlew'),
                    '--project-dir',
                    self._market_health_repository_dir,
                    ':health-integration-tests:assemble'
                ]
            )
        except sdk2.helpers.ProcessLog.CalledProcessError as e:
            raise self.IntegrationTestsError(e)

    @staticmethod
    def _wait_for_start_clickhouse():
        docker_host = os.environ.get("DOCKER_HOSTNAME")
        if docker_host is None:
            docker_host = 'localhost'

        deadline = time.time() + 300.0
        logger.info("Checking clickhouse server availability. Docker hostname: %s", docker_host)
        while True:
            current_time = time.time()

            time_left = deadline - current_time
            logger.info("Checking clickhouse server availability. Time left: %f sec.", time_left)
            if current_time >= deadline:
                raise Exception(
                    ("Timed out while waiting for 'clickhouse' with ip address {} to start. "
                     "Current_time {}, deadline {}").format(docker_host, current_time, deadline)
                )

            try:
                requests.get("http://{}:8123".format(docker_host), timeout=5.0)
                return
            except requests.exceptions.Timeout:
                logger.info("Checking clickhouse server availability: timeout")
                continue
            except requests.exceptions.RequestException as e:
                logger.info("Checking clickhouse server availability: exception %s", str(e))
                time.sleep(5)

    def _run_integration_tests(self):
        try:
            # Лог запишется в tests.out.log
            self._run_with_custom_logger(
                'tests',
                [
                    os.path.join(self._market_health_repository_dir, 'gradlew'),
                    '--project-dir',
                    self._market_health_repository_dir,
                    ':health-integration-tests:integrationTest',
                ]
            )
        except sdk2.helpers.ProcessLog.CalledProcessError as e:
            raise self.IntegrationTestsError(e)

    def _collect_debug_info_about_mount_points(self):
        self._run_ignoring_errors(['mount'])

    def _collect_debug_info_about_network(self):
        self._run_ignoring_errors(['ip', 'addr'])
        self._run_ignoring_errors(['ip', 'route'])

    def _collect_debug_info_about_docker(self):
        self._run_ignoring_errors(['systemctl', 'status', 'docker.service'])
        self._run_ignoring_errors(['journalctl', '-u', 'docker.service'])
        self._run_ignoring_errors(['docker', 'info'])

    def _run_ignoring_errors(self, args):
        try:
            self._run(args)
        except:
            pass

    def _run(self, args):
        # Лог запишется в common.log
        self._run_with_custom_logger(logging.getLogger(args[0]), args)

    def _run_with_custom_logger(self, logger, args):
        with sdk2.helpers.ProcessLog(self, logger=logger) as pl:
            # Так и задумано, stderr в stdout, чтобы всё сложилось в один файл, так удобнее.
            subprocess.check_call(args, stdout=pl.stdout, stderr=pl.stdout)

    @property
    def _market_health_repository_dir(self):
        return str(self.path('market-health'))

    @property
    def _docker_data_dir(self):
        return str(self.path('docker-data'))

    @staticmethod
    def _write_json_to_log_and_file(file, data):
        formatted_json = json.dumps(data, indent=4)
        logger.info('Writing to file %s: %s', formatted_json)
        file_utils.write_file(file, formatted_json)

    class IntegrationTestsError(common.errors.TaskFailure):
        def __init__(self, called_process_error):
            self.called_process_error = called_process_error

        def get_task_info(self):
            return (
                    "<br>"
                    "Интеграционные тесты упали. Скорее всего это значит что есть невалидные метрики.<br>\n"
                    "Открой tests.out.log (ссылка ниже) и поищи там ' ERROR ', большими буквами, с пробелами, без кавычек.<br>"
                    "<br>"
                    + self.called_process_error.get_task_info()
            )
