# coding=utf-8
import logging
import os
import io
from sandbox import sdk2
from sandbox.projects.common.arcadia import sdk
import sandbox.projects.common.constants as consts
from sandbox.projects.market.infra.MarketApplicationArcProducerBase import MarketApplicationArcProducerBase
import sandbox.projects.market.infra.helpers.app_producer as producer


LIQUIBASE_CHANGELOG_TEMPLATE = u"""
<databaseChangeLog
        xmlns="http://www.liquibase.org/xml/ns/dbchangelog"
        xmlns:xsi="http://www.w3.org/2001/XMLSchema-instance"
        xsi:schemaLocation="http://www.liquibase.org/xml/ns/dbchangelog
        http://www.liquibase.org/xml/ns/dbchangelog/dbchangelog-3.3.xsd">
</databaseChangeLog>
"""
LIQUIBASE_CHANGELOG_DEFAULT_PATH = "src/main/resources/liquibase/db-changelog.xml"
MODULE_PATH = "market/jlibrary/request/request-utils/src/main/java/ru/yandex/market/request/trace"
REVIEWS_URL = 'https://a.yandex-team.ru/review/'


class MarketJavaApplicationArcProducer(MarketApplicationArcProducerBase):
    class Parameters(MarketApplicationArcProducerBase.Parameters):
        source_dir = sdk2.parameters.String("Template java application dir in VCS",
                                            default='/market/infra/java-application/templates', required=False)
        ya_package_config_path = sdk2.parameters.String("Path to ya package config", required=False)
        dest_dir = sdk2.parameters.String("Destination dir in VCS", required=True)
        java_package = sdk2.parameters.String("Java package (ex: ru.yandex.market.team.application)", required=True)
        ya_owner = sdk2.parameters.String("Owner for ya.make (ex: g:marketinfra)", required=True)
        author = sdk2.parameters.String("Spok author / maintainer of service", required=True)
        create_liquibase_changelog = sdk2.parameters.Bool("Add liquibase changelog.xml file", default=False)
        template_name = sdk2.parameters.String("Template name", default='generic-template', required=False)
        module_name = sdk2.parameters.String("Module name to trace requests from a service", default="", required=False)
        deploy_type = sdk2.parameters.String("Deploy type", default="yandex_deploy", required=False)

    @property
    def base_branch_name(self):
        return 'MarketJavaApplicationArcProducer'

    def produce_application(self):
        logging.info("Starting MarketJavaApplicationArcProducer task with params: %s", self.Parameters)

        source_app_dir = self.Parameters.source_dir.strip('/')
        dest_app_dir = self.Parameters.dest_dir.strip('/')

        template_dir = os.path.join(self.arcadia_root, source_app_dir, self.Parameters.template_name)
        arcadia_dir = os.path.join(self.arcadia_root, dest_app_dir)

        if self.Parameters.ya_package_config_path:
            package_json_path = os.path.join(self.arcadia_root, self.Parameters.ya_package_config_path)
        else:
            package_json_path = os.path.join(arcadia_dir, 'package.json')

        if os.path.exists(arcadia_dir):
            if producer.is_package_json_for_current_app(self.Parameters.application_name, package_json_path):
                logging.info('Code for this App already generated. Do nothing')
                return False
            else:
                raise RuntimeError('Path "{}" already exists'.format(arcadia_dir))

        producer.produce_application(
            template_dir,
            self.arcadia_root,
            dest_app_dir,
            self.Parameters.application_name,
            self.Parameters.java_package,
            self.Parameters.ya_owner,
            self.Parameters.author,
            self.Parameters.template_name,
            self.Parameters.module_name,
            self.Parameters.deploy_type,
            package_json_path
        )

        self.add_trace_module(self.arcadia_root)
        self.add_liquibase_changelog(arcadia_dir)
        self.build_app(self.arcadia_root, dest_app_dir)

        logging.info("Do package")
        sdk.do_package(self.arcadia_root, ["%s/package.json" % dest_app_dir], clear_build=False, checkout=True, direct_arcadia_mode=True)

        logging.info('Application created!')

        return True

    def build_app(self, arcadia_root, dest_app_dir):
        # Run build after all svn files was added, cause it will create some garbage
        logging.info("Running build")
        sdk.do_build(
            consts.YMAKE_BUILD_SYSTEM, arcadia_root, ['market/jlibrary/request/request-utils'],
            test=False, checkout=True, clear_build=False, build_threads=0
        )
        sdk.do_build(
            consts.YMAKE_BUILD_SYSTEM, arcadia_root, [dest_app_dir], test=True, checkout=True, clear_build=False
        )

    def add_liquibase_changelog(self, arcadia_dir):
        # Add liquibase changelog file to default location for liquibase job
        if self.Parameters.create_liquibase_changelog:
            liquibase_changelog_path = os.path.join(arcadia_dir, LIQUIBASE_CHANGELOG_DEFAULT_PATH)
            os.makedirs(os.path.dirname(liquibase_changelog_path))

            with io.open(liquibase_changelog_path, mode="w", encoding="utf-8") as f:
                f.write(LIQUIBASE_CHANGELOG_TEMPLATE)

    def add_trace_module(self, arcadia_root):
        module_name = self.Parameters.module_name
        if not module_name:
            return

        module_name = module_name.upper().replace("-", "_")
        module_file_path = os.path.join(arcadia_root, MODULE_PATH, 'Module.java')

        logging.info("Adding '%s' module  to '%s'", module_name, module_file_path)

        with io.open(module_file_path, mode="r", encoding="utf-8") as f:
            content = f.read()

        new_content = ''
        for line in content.splitlines():
            new_content = new_content + line + '\n'

            if 'enum' in line and 'Module' in line:
                extra_line_with_module = '    %s("%s"),' % (module_name, module_name.lower())
                new_content = new_content + extra_line_with_module + '\n'

        with io.open(module_file_path, mode="w", encoding="utf-8") as f:
            f.write(new_content)
