# coding=utf-8
import logging
from sandbox import sdk2
from sandbox.sdk2.vcs.svn import Arcadia
import os
import io
from sandbox.projects.common.arcadia import sdk
import sandbox.projects.common.constants as consts
import sandbox.projects.market.infra.helpers.app_producer as producer
import sandbox.projects.market.infra.helpers.arcadia as arcadia_helpers


class MarketJavaApplicationProducer(sdk2.Task):
    class Parameters(sdk2.Task.Parameters):
        source_dir = sdk2.parameters.String("Template java application dir in VCS",
                                            default='/market/infra/java-application/templates', required=False)
        dest_dir = sdk2.parameters.String("Destination dir in VCS", required=True)
        ya_package_config_path = sdk2.parameters.String("Path to ya package config", required=False)
        java_package = sdk2.parameters.String("Java package (ex: ru.yandex.market.team.application)", required=True)
        application_name = sdk2.parameters.String("Application name (ex: market-team-application)", required=True)

        ya_owner = sdk2.parameters.String("Owner for ya.make (ex: g:marketinfra)", required=True)
        author = sdk2.parameters.String("Spok author / maintainer of service", required=True)
        commit_message = sdk2.parameters.String("Commit message for SVN", required=True)
        arcadia_user = sdk2.parameters.String('Arcadia user', default='zomb-sandbox-rw')
        create_liquibase_changelog = sdk2.parameters.Bool("Add liquibase changelog.xml file", default=False)
        template_name = sdk2.parameters.String("Template name", default='generic-template', required=False)
        module_name = sdk2.parameters.String("Module name to trace requests from a service", default="", required=False)
        deploy_type = sdk2.parameters.String("Deploy type", default="yandex_deploy", required=False)

    def on_execute(self):
        logging.info("Starting MarketJavaApplicationProducer task with params: %s", self.Parameters)

        template_dir = Arcadia.checkout(
            'arcadia:/arc/trunk/arcadia/' + self.Parameters.source_dir + '/' + self.Parameters.template_name,
            path=str(self.path("template"))
        )

        dest_app_dir = self.Parameters.dest_dir.strip('/')

        arcadia_dir = arcadia_helpers.checkout_immediates(str(self.path("arcadia")), dest_app_dir)

        dest_app_dir_abs = os.path.join(arcadia_dir, dest_app_dir)

        if self.Parameters.ya_package_config_path:
            package_json_path = os.path.join(self.arcadia_root, self.Parameters.ya_package_config_path)
        else:
            package_json_path = os.path.join(arcadia_dir, 'package.json')
        if os.path.exists(dest_app_dir_abs):
            if producer.is_package_json_for_current_app(self.Parameters.application_name, package_json_path):
                logging.info('Code for this App already generated. Do nothing')
                return
            else:
                raise RuntimeError('Path "{}" already exists'.format(dest_app_dir_abs))

        producer.produce_application(
            template_dir,
            arcadia_dir,
            dest_app_dir,
            self.Parameters.application_name,
            self.Parameters.java_package,
            self.Parameters.ya_owner,
            self.Parameters.author,
            self.Parameters.template_name,
            self.Parameters.module_name,
            self.Parameters.deploy_type,
            package_json_path
        )

        self.add_trace_module(arcadia_dir)
        self.add_liquibase_changelog(arcadia_dir)

        arcadia_helpers.safe_add_files(arcadia_dir)

        # Run build after all svn files was added, cause it will create some garbage
        logging.info("Running build")
        sdk.do_build(
            consts.YMAKE_BUILD_SYSTEM, arcadia_dir, ['market/jlibrary/request/request-utils'],
            test=False, checkout=True, clear_build=False, build_threads=0
        )
        sdk.do_build(
            consts.YMAKE_BUILD_SYSTEM, arcadia_dir, [dest_app_dir], test=True, checkout=True, clear_build=False
        )
        # do_package adds 'arcadia' to source root, but do_build don't. C - consistency.
        ya_package_source_root = os.path.dirname(arcadia_dir)

        logging.info("Do package")
        sdk.do_package(ya_package_source_root, ["%s/package.json" % dest_app_dir], clear_build=False, checkout=True)

        logging.info('Commiting to arcadia with message: %s', self.Parameters.commit_message)
        Arcadia.commit(arcadia_dir, self.Parameters.commit_message, self.Parameters.arcadia_user)
        logging.info('Application created!')

    def add_liquibase_changelog(self, arcadia_dir):

        LIQUIBASE_CHANGELOG_TEMPLATE = u"""
        <databaseChangeLog
                xmlns="http://www.liquibase.org/xml/ns/dbchangelog"
                xmlns:xsi="http://www.w3.org/2001/XMLSchema-instance"
                xsi:schemaLocation="http://www.liquibase.org/xml/ns/dbchangelog
                http://www.liquibase.org/xml/ns/dbchangelog/dbchangelog-3.3.xsd">
        </databaseChangeLog>
        """
        LIQUIBASE_CHANGELOG_DEFAULT_PATH = "src/main/resources/liquibase/db-changelog.xml"

        # Add liquibase changelog file to default location for liquibase job
        if self.Parameters.create_liquibase_changelog:
            liquibase_changelog_path = os.path.join(
                arcadia_dir,
                self.Parameters.dest_dir,
                LIQUIBASE_CHANGELOG_DEFAULT_PATH
            )
            os.makedirs(os.path.dirname(liquibase_changelog_path))

            with io.open(liquibase_changelog_path, mode="w", encoding="utf-8") as f:
                f.write(LIQUIBASE_CHANGELOG_TEMPLATE)

    def add_trace_module(self, arcadia_dir):
        module_name = self.Parameters.module_name
        if not module_name:
            return

        module_name = module_name.upper().replace("-", "_")

        module_path = 'market/jlibrary/request/request-utils/src/main/java/ru/yandex/market/request/trace'
        module_temp_dir = Arcadia.checkout(
            'arcadia:/arc/trunk/arcadia/' + module_path, path=str(self.path("module_temp")), depth='files'
        )
        arcadia_dir = arcadia_helpers.update_local_path(arcadia_dir, module_path)

        src_module_file = os.path.join(module_temp_dir, 'Module.java')
        dest_module_file = os.path.join(arcadia_dir, module_path.strip('/'), 'Module.java')

        logging.info("Copying trace module file with adding '%s' module from '%s' to '%s'", module_name,
                     src_module_file, dest_module_file)

        with io.open(src_module_file, mode="r", encoding="utf-8") as f:
            content = f.read()

        new_content = ''
        for line in content.splitlines():
            new_content = new_content + line + '\n'

            if 'enum' in line and 'Module' in line:
                extra_line_with_module = '    %s("%s"),' % (module_name, module_name.lower())
                new_content = new_content + extra_line_with_module + '\n'

        with io.open(dest_module_file, mode="w", encoding="utf-8") as f:
            f.write(new_content)
