# coding=utf-8
import logging
import os
import sandbox.projects.common.constants as consts
import shutil
import subprocess
from sandbox import sdk2
from sandbox.projects.common.arcadia import sdk
from sandbox.sdk2.vcs.svn import Arcadia


class MarketPublishJavaLibrary(sdk2.Task):
    class Requirements(sdk2.Task.Requirements):
        container_resource = sdk2.parameters.Container(default=773239891)

    class Parameters(sdk2.Task.Parameters):
        ya_make_dir = sdk2.parameters.String(
            "Dir with ya.make. Can be directly java library or ya.make with recurse", required=True
        )
        repository_url = sdk2.parameters.String(
            "Maven repo user. Ex. http://artifactory.yandex.net/artifactory/yandex_market_releases/", required=True
        )
        repository_user = sdk2.parameters.String("Maven repo user (and password for now)", required=True)
        arcadia_revision = sdk2.parameters.String("Arcadia revision", default=None, required=False)
        jdk_version = sdk2.parameters.Integer("JDK version", default=8, required=False)

    def on_execute(self):
        logging.info("Starting MarketPublishJavaLibrary task with params: %s", self.Parameters)

        m2_tar_gz = os.path.join(os.getcwd(), 'm2.tar.gz')
        m2_dir = os.path.join(os.getcwd(), 'm2')

        rc = subprocess.call(
            ['wget', '-O', m2_tar_gz, 'http://proxy.sandbox.yandex-team.ru/80669749'],
            stdout=subprocess.PIPE,
            stderr=subprocess.PIPE
        )
        logging.info('wget: %s', 'FAIL' if rc else 'OK')

        if os.path.exists(m2_dir):
            shutil.rmtree(m2_dir)
        os.mkdir(m2_dir)

        subprocess.call(
            ['tar', '-xzf', m2_tar_gz, '-C', m2_dir],
            stdout=subprocess.PIPE,
            stderr=subprocess.PIPE
        )
        logging.info('tar: %s', 'FAIL' if rc else 'OK')

        maven_settings = """
<settings>
    <localRepository>%s</localRepository>
    <servers>
        <server>
            <id>%s</id>
            <username>%s</username>
            <password>%s</password>
        </server>
    </servers>
</settings>""" % (
            m2_dir, self.Parameters.repository_user, self.Parameters.repository_user, self.Parameters.repository_user
        )

        logging.info("Maven settings: /n%s" % maven_settings)

        setting_file = os.path.join(os.getcwd(), 'settings.xml')
        with open(setting_file, 'w') as settings:
            settings.write(maven_settings)

        ya_make_rel_dir = self.Parameters.ya_make_dir.strip('/')

        svn_rev = self.Parameters.arcadia_revision if self.Parameters.arcadia_revision is not None else Arcadia.get_revision("arcadia:/arc/trunk/arcadia")
        logging.info("svn_rev = %s" % svn_rev)

        with sdk.mount_arc_path("arcadia-arc:/#r" + svn_rev) as arcadia_dir:
            ya_make_absolute_dir = os.path.join(arcadia_dir, ya_make_rel_dir)

            logging.info("Running build with tests in dir: %s " % ya_make_absolute_dir)
            sdk.do_build(
                consts.YMAKE_BUILD_SYSTEM, arcadia_dir, [ya_make_rel_dir], test=True, clear_build=True
            )

            ya_path = os.path.join(arcadia_dir, 'ya')
            ya_command = [
                ya_path,
                'make',
                '--maven-export',
                '--deploy',
                '--repository-id',
                self.Parameters.repository_user,
                '--repository-url',
                self.Parameters.repository_url,
                '--settings',
                setting_file,
                "--version",
                svn_rev,
                '--sources',
                '--verbose',
                '--rebuild',
                '-DJDK_VERSION=%d' % self.Parameters.jdk_version,
                ya_make_absolute_dir
            ]

            logging.info("Running maven-export command: %s" % ya_command)

            java_bin = subprocess.check_output([ya_path, 'tool', 'java', '--print-path'])
            java_home = os.path.normpath(os.path.join(java_bin, '..', '..', 'jre'))

            env = os.environ.copy()
            env['JAVA_HOME'] = java_home

            process = subprocess.Popen(
                ya_command,
                stderr=subprocess.PIPE,
                stdout=subprocess.PIPE,
                env=env
            )

            out, err = process.communicate()
            return_code = process.wait()

        logging.info('Out: ' + str(out))
        logging.info('Err: ' + str(err))
        logging.info('Return code: %s', return_code)

        if return_code:
            raise Exception('Ya make command failed')
