# coding=utf-8
import logging
from sandbox import sdk2
import os
import io
from shutil import copytree, ignore_patterns
import jinja2
from sandbox.projects.common.arcadia import sdk
import sandbox.projects.common.constants as consts
from sandbox.projects.market.infra.MarketApplicationArcProducerBase import MarketApplicationArcProducerBase
import sandbox.projects.market.infra.helpers.app_producer as producer


class MarketPythonApplicationArcProducer(MarketApplicationArcProducerBase):
    class Parameters(MarketApplicationArcProducerBase.Parameters):
        source_dir = sdk2.parameters.String("Template python application dir in Arcadia",
                                            default='/market/devexp/app_templates/python_app/', required=False)
        dest_dir = sdk2.parameters.String("Destination dir in Arcadia", required=True)
        ya_owner = sdk2.parameters.String("Owner for ya.make (ex: g:marketinfra)", required=True)
        maintainer = sdk2.parameters.String("Spok author / maintainer of service", required=True)
        ya_package_config_path = sdk2.parameters.String("Path to ya package config", required=False)

    @property
    def base_branch_name(self):
        return 'MarketPythonApplicationArcProducer'

    def produce_application(self):
        logging.info("Starting MarketPythonApplicationArcProducer task with params: %s", self.Parameters)

        source_dir = self.Parameters.source_dir.strip('/')
        template_dir = os.path.join(self.arcadia_root, source_dir)
        dest_app_dir = self.Parameters.dest_dir.strip('/')
        dest_dir = os.path.join(self.arcadia_root, dest_app_dir)

        if self.Parameters.ya_package_config_path:
            package_json_path = os.path.join(self.arcadia_root, self.Parameters.ya_package_config_path)
        else:
            package_json_path = os.path.join(dest_dir, 'pkg.json')
        if os.path.exists(dest_dir):
            if producer.is_package_json_for_current_app(self.Parameters.application_name, package_json_path):
                logging.info('Code for this App already generated. Do nothing')
                return False
            else:
                raise RuntimeError('Path "{}" already exists'.format(dest_dir))

        self.prepare_app(template_dir, dest_dir, self.Parameters)

        producer.process_parent_ya_make(dest_dir, self.Parameters.ya_owner)

        logging.info("Running build")
        bin_dest_app_dir = os.path.join(dest_app_dir, "bin")
        sdk.do_build(
            consts.YMAKE_BUILD_SYSTEM, self.arcadia_root, [bin_dest_app_dir],
            test=True, checkout=True, clear_build=False
        )

        logging.info("Do package")
        sdk.do_package(
            self.arcadia_root, ["%s/pkg.json" % dest_app_dir],
            clear_build=False, checkout=True, convert=True, direct_arcadia_mode=True
        )

        logging.info('Application created!')

        return True

    def prepare_app(self, copy_from, copy_to, params):
        copytree(copy_from, copy_to, ignore=ignore_patterns(".svn"))

        for root, folders, files in os.walk(copy_to):
            for file_name in files:
                abs_filepath = os.path.join(root, file_name)

                filename_without_ext, file_extension = os.path.splitext(file_name)
                new_path = os.path.join(root, filename_without_ext)

                if file_extension == ".jinja2":
                    logging.info("Processing jinja2 template file: %s", abs_filepath)
                    logging.info("Copying template to: %s", new_path)

                    with io.open(abs_filepath, "r", encoding="utf-8") as file:
                        tmpl = file.read()

                    with io.open(new_path, "w", encoding="utf-8") as file:
                        template_context = {
                            "app_name": params.application_name,
                            "app_path": self.Parameters.dest_dir.strip('/'),
                            "maintainer": params.maintainer,
                            "ya_owner": params.ya_owner
                        }

                        tmpl_processed = jinja2.Template(tmpl, keep_trailing_newline=True).render(template_context)
                        file.write(tmpl_processed)

                    logging.info("Removing old template: %s", abs_filepath)
                    os.remove(abs_filepath)

                if root == copy_to and filename_without_ext == "pkg.json" and params.ya_package_config_path:
                    ya_package_config_name = os.path.basename(params.ya_package_config_path)
                    ya_package_config_dir = os.path.join(self.arcadia_root,
                                                         os.path.dirname(params.ya_package_config_path))
                    if copy_to != ya_package_config_dir or ya_package_config_name != "pkg.json":
                        os.replace(os.path.join(root, "pkg.json"), os.path.join(self.arcadia_root,
                                                                                params.ya_package_config_path))
