import os
import sys
import string
import logging
import textwrap

from sandbox import sdk2
from sandbox.sdk2.vcs.svn import Arcadia

from sandbox.projects.common.arcadia import sdk as arcadiasdk


class Pathfinder(object):
    def __init__(self, path):
        self.__path = path

    def __enter__(self):
        sys.path.insert(0, self.__path)

    def __exit__(self, *_):
        sys.path.remove(self.__path)


class MarketResourceTypeProducer(sdk2.Task):
    """Produce a new Resource Type. Only for Market services.
    """

    class Parameters(sdk2.Task.Parameters):
        resource_type = sdk2.parameters.String("Resource Type", required=True)
        ya_owner = sdk2.parameters.String("Owner for ya.make (ex: g:marketsre)", default='g:marketsre')
        commit_message = sdk2.parameters.String("Commit message for SVN", required=True)
        arcadia_user = sdk2.parameters.String('Arcadia user', default='zomb-sandbox-rw')
        fail_on_exists = sdk2.parameters.Bool('Fail on Resource Type exists', required=False, default=True)

    def on_execute(self):
        logging.info("Staring %s task with params: %s", self.type, self.Parameters)

        if not self._is_resource_type_valid(resource_type=self.Parameters.resource_type):
            raise ValueError('Resource type name {} is invalid'.format(self.Parameters.resource_type))

        if self._resource_type_exists(resource_type=self.Parameters.resource_type):
            if self.Parameters.fail_on_exists:
                error_msg = 'Resource type {} already exists'.format(self.Parameters.resource_type)
                logging.error(error_msg)
                raise Exception(error_msg)
            else:
                logging.info('Resource type {} already exists'.format(self.Parameters.resource_type))
                return

        resources_dir = Arcadia.checkout(
            url="arcadia:/arc/trunk/arcadia/sandbox/projects/market/resources",
            path=str(self.path("resources")),
        )
        resource_type_camel_case = self._resource_type_to_camel_case(self.Parameters.resource_type)
        resource_type_dir = os.path.join(resources_dir, "generated", resource_type_camel_case)

        if not self._is_resource_type_class_valid(resource_type_camel_case):
            raise ValueError('Resource type class name {} is invalid'.format(resource_type_camel_case))

        if os.path.exists(resource_type_dir):
            if self.Parameters.fail_on_exists:
                error_msg = 'Resource type directory {} already exists'.format(resource_type_dir)
                logging.error(error_msg)
                raise Exception(error_msg)
            else:
                logging.info('Resource type directory {} already exists'.format(resource_type_dir))
                return

        logging.info('Producing resource type %s (%s)', self.Parameters.resource_type, resource_type_dir)
        self._produce_resource_type(
            resource_type_dir=resource_type_dir,
            resource_type_class=self.Parameters.resource_type,
        )

        with arcadiasdk.mount_arc_path("arcadia:/arc/trunk/arcadia/devtools/ya") as devtools_lib:
            with Pathfinder(devtools_lib):
                self.patch_ya_make(resources_dir, resource_type_dir)

        logging.info('Adding directory %s', resource_type_dir)
        Arcadia.add(resource_type_dir)

        logging.info('Commiting to arcadia with message: %s', self.Parameters.commit_message)
        Arcadia.commit(resources_dir, self.Parameters.commit_message, self.Parameters.arcadia_user)
        logging.info('Resource type has been committed, it will be ready to use soon')

    def patch_ya_make(self, resources_dir, resource_type_dir):
        """
        Acting on assumption that resources are stored in sandbox/projects/market/resources/generated,
        do the following:
            - add source file to sandbox/projects/market/resources/ya.make, where they belong to as of now;
            - add a ya.make with OWNER() macro to a freshly generated resource type's directory
        """

        import yalibrary.makelists

        root_ya_make = os.path.join(resources_dir, "ya.make")
        new_source_path = os.path.join(os.path.relpath(resource_type_dir, resources_dir), "__init__.py")
        logging.debug("Adding %s to %s", new_source_path, root_ya_make)
        yamake = yalibrary.makelists.from_file(root_ya_make)
        sources_node = next(iter(yamake.find_siblings("PY_SRCS")), None)
        assert sources_node

        assert new_source_path not in (_.name for _ in sources_node.children)
        sources_node.append_child(yalibrary.makelists.macro_definitions.SrcValue(new_source_path))
        yamake.write(root_ya_make)

        module_ya_make_path = os.path.join(resource_type_dir, "ya.make")
        logging.info("Writing %s", module_ya_make_path)
        with open(module_ya_make_path, "w") as fp:
            fp.write("OWNER({owners})\n".format(owners=self.Parameters.ya_owner))

    @staticmethod
    def _resource_type_exists(resource_type):
        """Check the resource type exists.
        """
        return resource_type in sdk2.Resource

    @staticmethod
    def _resource_type_to_camel_case(resource_type):
        """Convert the string in snake case into camel case.
        """
        return ''.join(s.title() for s in resource_type.split('_'))

    @staticmethod
    def _produce_resource_type(resource_type_dir, resource_type_class):
        """Produce a new resource type:
        - create the resource type directory
        - add the directory into the parent ya.make
        """
        content = textwrap.dedent("""
            # coding=utf-8
            from sandbox import sdk2


            class {resource_type_class}(sdk2.Resource):
                releasable = True
                releasers = ["MARKET"]
                auto_backup = True
        """).format(resource_type_class=resource_type_class).strip()

        os.mkdir(resource_type_dir)
        with open(os.path.join(resource_type_dir, '__init__.py'), 'w') as fp:
            fp.write(content + '\n')

    @staticmethod
    def _create_ya_make(ya_make_path, owners):
        """Create a new ya.make file.
        """

    @staticmethod
    def _is_resource_type_class_valid(resource_type_camelcase):
        """Validate the resource type class name. It must contain only letters or/and digits and starts from a letter.
        """
        if not resource_type_camelcase:
            return False

        if resource_type_camelcase[0] not in string.ascii_letters:
            return False

        allowed_symbols = string.ascii_letters + string.digits
        return all(s in allowed_symbols for s in resource_type_camelcase)

    @staticmethod
    def _is_resource_type_valid(resource_type):
        """Validate the resource type name. It must contain only uppercase letters or/and digits.
        """
        if not resource_type:
            return False

        allowed_symbols = string.ascii_uppercase + string.digits + '_'
        return all(s in allowed_symbols for s in resource_type)
