import io
import json
import logging
import os
import shutil


def ignored_files(src, names):
    return {
        'SpringApplicationConfig.java',
        'ApplicationTemplateMain.java',
        'ru',
        'ya.make',
        'package.json',
        '.svn',
        'properties.d',
        '.devexp.json'
    }


replacements = {}
filename_replacements = {}


def produce_application(source_dir, dest_repo_dir, dest_app_dir, application_name, java_package, ya_owner, author,
                        template_name, module_name, deploy_type, ya_package_config_path=None,
                        rewrite_existing_dir_if_empty=False):
    """
    :type source_dir: str
    :type dest_repo_dir: str
    :type dest_app_dir: str
    :type java_package: str
    :type application_name: str
    :type ya_owner: str
    :type author: str
    :type template_name: str
    :type module_name: str
    :type deploy_type: str
    :type ya_package_config_path: str
    :type rewrite_existing_dir_if_empty bool
    """

    dest_app_dir = dest_app_dir.strip('/')
    dest_dir = os.path.join(dest_repo_dir, dest_app_dir)
    logging.info(
        "Creating application '%s' in dir '%s'.\n Template name: '%s', template path: '%s'.\n" +
        "Java package '%s'.\nOwner '%s'.\nAuthor: '%s'",
        application_name, dest_dir, template_name, source_dir, java_package, ya_owner, author
    )

    if os.path.exists(dest_dir):
        if rewrite_existing_dir_if_empty is True and len(os.listdir(dest_dir)) == 0:
            shutil.rmtree(dest_dir)
        else:
            raise Exception("Destination dir %s already exists." % dest_dir)

    # copy as is, but ignore some files, (some of them we will copy in custom way below)
    shutil.copytree(source_dir, dest_dir, False, ignored_files)

    main_class_name = application_name.title().replace('-', '')

    # correct template names: [
    # generic-template, generic-with-postgres-template, bazinga-tms-with-mongo-template, quartz-tms-template,
    # mj-template
    # ]

    replacements['market-java-application-template'] = application_name
    replacements['mj-application-template'] = application_name
    replacements['market_java_framework_template'] = application_name
    replacements['market-java-application-' + template_name] = application_name
    replacements['market/infra/java-application/templates/' + template_name] = dest_app_dir
    replacements['g:marketinfra'] = ya_owner
    replacements['robot-market-infra'] = author
    replacements['ApplicationTemplateMain'] = main_class_name
    replacements['ru.yandex.market.template'] = java_package

    if template_name == 'mj-template' and deploy_type == 'nanny':
        replacements['market/infra/java-application/core/package-include-deploy.json'] =\
            'market/infra/java-application/core/package-include.json'
        replacements['/app/service.yaml'] = '/{package_name}/service.yaml'
        replacements['deploy_type: yandex_deploy'] = 'deploy_type: nanny'

    filename_replacements['ApplicationTemplateMain'] = main_class_name

    src_main_java = '/src/main/java/'
    src_test_java = '/src/test/java/'

    copy_subdirs_with_replace(
        source_dir + src_main_java + 'ru/yandex/market/template',
        dest_dir + src_main_java + java_package.replace('.', '/'),
        '.java'
    )
    copy_subdirs_with_replace(
        source_dir + src_test_java + 'ru/yandex/market/template',
        dest_dir + src_test_java + java_package.replace('.', '/'),
        '.java'
    )
    copy_subdirs_with_replace(
        source_dir + '/src/main/properties.d/', dest_dir + '/src/main/properties.d/', '.properties'
    )

    copy_with_replace(source_dir, dest_dir, 'src/main/conf/log4j2.xml')
    copy_with_replace(source_dir, dest_dir, 'src/main/external/push-client.conf')
    copy_with_replace(source_dir, dest_dir, 'src/test/ya.make')
    copy_with_replace(source_dir, dest_dir, 'ya.make')

    if ya_package_config_path:
        ya_package_config_name = os.path.basename(ya_package_config_path)
        ya_package_config_dir = os.path.dirname(ya_package_config_path)
        copy_and_rename_with_replace(source_dir, ya_package_config_dir, 'package.json', ya_package_config_name)
    else:
        copy_with_replace(source_dir, dest_dir, 'package.json')

    replace_trace_module_markers(
        '/config/SpringApplicationConfig.java',
        source_dir + src_main_java + 'ru/yandex/market/template',
        dest_dir + src_main_java + java_package.replace('.', '/'),
        module_name
    )
    replace_trace_module_markers(
        '/service.yaml',
        source_dir,
        dest_dir,
        module_name
    )
    process_parent_ya_make(dest_dir, ya_owner)


def replace_trace_module_markers(file_to_replace, source_dir, dest_dir, module_name):
    dest_file = dest_dir + file_to_replace
    source_file = source_dir + file_to_replace

    if not os.path.isfile(source_file):
        logging.info("There is no file " + source_file + " in template. Skipping")
        return

    exclude_lines = []

    if not module_name:
        exclude_lines = ['//REPLACEMENT_MARKER_MODULE_IMPORT']
        replacements['/* REPLACEMENT_MARKER_MODULE_NAME */'] = ''
    else:
        replacements['//REPLACEMENT_MARKER_MODULE_IMPORT'] = 'import ru.yandex.market.request.trace.Module;'
        replacements['null/* REPLACEMENT_MARKER_MODULE_NAME */'] = 'Module.' + module_name
        replacements['# REPLACEMENT_MARKER_MODULE_NAME'] = module_name

    do_copy_with_replace_and_exclude(source_file, dest_file, exclude_lines)


def recursive_copy_with_replace(full_src, full_dst, ext):
    for path, _, files in os.walk(full_src):
        for file in files:
            if ext in file:
                new_filename = file
                for replacement in filename_replacements:
                    if replacement in file:
                        new_filename = file.replace(replacement, filename_replacements[replacement])

                logging.info("Copying file: from '%s' to '%s'", path + '/' + file,
                             path.replace(full_src, full_dst) + '/' + new_filename)
                do_copy_with_replace(path + '/' + file, path.replace(full_src, full_dst) + '/' + new_filename)


def copy_config_files(source_dir, dest_dir, java_package, filename):
    copy_with_replace(
        source_dir + '/src/main/java/ru/yandex/market/template/config',
        dest_dir + '/src/main/java/' + java_package.replace('.', '/') + '/config',
        filename
    )


def copy_subdirs_with_replace(source_dir, dest_dir, ext):
    """
    :type source_dir: str
    :type dest_dir: str
    :type ext: str
    """
    recursive_copy_with_replace(source_dir, dest_dir, ext)


def copy_main(source_dir, dest_dir, java_package, main_class_name):
    """
    :type source_dir: str
    :type dest_dir: str
    :type java_package: str
    :type main_class_name: str
    """

    main_class_dir = java_package.replace('.', '/')
    main_class_path = os.path.join(dest_dir, 'src/main/java', main_class_dir, main_class_name + '.java')
    test_class_path = os.path.join(dest_dir, 'src/test/java', main_class_dir, main_class_name + 'Test.java')

    do_copy_with_replace(
        os.path.join(source_dir, 'src/main/java/ru/yandex/market/template/ApplicationTemplateMain.java'),
        main_class_path
    )

    do_copy_with_replace(
        os.path.join(source_dir, 'src/test/java/ru/yandex/market/template/ApplicationTemplateMainTest.java'),
        test_class_path
    )


def copy_with_replace(source_dir, dest_dir, file_path):
    copy_and_rename_with_replace(source_dir, dest_dir, file_path, file_path)


def copy_and_rename_with_replace(source_dir, dest_dir, source_file_path, dest_file_path):
    """
    :type source_dir: str
    :type dest_dir: str
    :type source_file_path: str
    :type dest_file_path: str
    """
    do_copy_with_replace(
        os.path.join(source_dir, source_file_path.lstrip('/')),
        os.path.join(dest_dir, dest_file_path.lstrip('/'))
    )


def do_copy_with_replace(source_path, dest_path):
    do_copy_with_replace_and_exclude(source_path, dest_path, [])


def do_copy_with_replace_and_exclude(source_path, dest_path, exclude_lines):
    """
    :type source_path: str
    :type dest_path: str
    :type exclude_lines: list
    """

    logging.info("Copying from %s to %s with replacements: %s", source_path, dest_path, replacements)

    with io.open(source_path, 'r', encoding='utf-8') as source_file:
        text = source_file.read()

    for replacement in replacements:
        text = text.replace(replacement, replacements[replacement])

    new_text = ""
    for line in text.splitlines():
        if any(e in line for e in exclude_lines):
            continue
        new_text += line + '\n'

    text = new_text

    if not os.path.exists(os.path.dirname(dest_path)):
        os.makedirs(os.path.dirname(dest_path))

    with io.open(dest_path, 'w', encoding='utf-8') as dest_file:
        dest_file.write(text)


def process_parent_ya_make(ya_make_dir, owner):
    """
    :type ya_make_dir: str
    :type owner: str
    """
    ya_make_dir = ya_make_dir.rstrip('/')
    parent_ya_make_dir = os.path.dirname(ya_make_dir)
    new_module_name = os.path.basename(ya_make_dir)
    ya_make_path = os.path.join(parent_ya_make_dir, 'ya.make')

    if os.path.exists(ya_make_path):
        add_module_to_ya_make(ya_make_path, new_module_name)
    else:
        create_new_ya_make(ya_make_path, new_module_name, owner)
        process_parent_ya_make(parent_ya_make_dir, owner)


def create_new_ya_make(ya_make_path, module_name, owner):
    ya_make_text = u'OWNER(%s)\n' \
                   '\n' \
                   'RECURSE(\n' \
                   '    %s\n' \
                   ')\n' % (owner, module_name)
    logging.info("Creating ya make file %s with content:\n%s" % (ya_make_path, ya_make_text))

    with io.open(ya_make_path, 'w', encoding='utf-8') as ya_make_file:
        ya_make_file.write(ya_make_text)


def add_module_to_ya_make(ya_make_path, new_module_name):
    logging.info("Adding module %s to ya make file %s", new_module_name, ya_make_path)
    with io.open(ya_make_path, 'r', encoding='utf-8') as ya_make_file:
        ya_make_text = ya_make_file.read()
    logging.info('Old ya make text:\n%s', ya_make_text)
    recurse_pos = ya_make_text.find('RECURSE')
    if recurse_pos >= 0:
        comma_pos = ya_make_text.index(')', recurse_pos)
        ya_make_text = '%s    %s\n%s' % (ya_make_text[:comma_pos], new_module_name, ya_make_text[comma_pos:])
    else:
        ya_make_text = '%s\nRECURSE(\n    %s\n)\n' % (ya_make_text, new_module_name)

    with io.open(ya_make_path, 'w', encoding='utf-8') as ya_make_file:
        ya_make_file.write(ya_make_text)
    logging.info('New ya make text:\n%s', ya_make_text)


def is_package_json_for_current_app(app_name, package_json_path):
    if os.path.exists(package_json_path):
        logging.info('Found file {} in app dir. Assuming that it\'s a package_json config. Trying to load',
                     package_json_path)
        with open(package_json_path, 'r') as fp:
            package_json_content = None
            try:
                package_json_content = json.load(fp)
            except:
                logging.info('Failed to load file {}. It\'s not a package_json config', package_json_path)
                return False

            if not package_json_content or not isinstance(package_json_content, dict):
                logging.info('File content has unexpected format. It\'s not a package_json config',
                             package_json_path)
                return False

            name_from_package_json = package_json_content.get("meta", {}).get("name", None)

            if app_name == name_from_package_json:
                return True

    return False
