# coding=utf-8
import json
import logging
import os
import re

from sandbox.sdk2.path import Path


class SvnLogHelper(object):
    # TODO: to come up with universal branch naming strategy
    PATH_PATTERN = '^((/branches/market/[\d|\.]+/)|' \
                   '(/trunk/)|(/branches/market/idx/\w+/[\d|\.]+/)|' \
                   '(/branches/junk/market/infra/[\w\d\.]+/))'

    def __get_logger(self, name):
        __logger = logging.getLogger(name)
        __logger.setLevel(logging.INFO)
        __logger.propagate = not name
        __logger.manager.loggerDict.pop(__logger.name)
        if self.log_path:
            fh = logging.FileHandler(str(Path(self.log_path).joinpath(name)))
            __logger.addHandler(fh)
        return __logger

    def __all_files_paths(self, files_black_list):
        logger = self.__get_logger('files_of_module')
        files = set()
        for root, _, file_names in os.walk(self.root_path):
            for filename in file_names:
                current_path = str(Path(root).joinpath(filename))
                current_path = current_path[len(self.root_path) + 1:]
                if not any(
                        bad_path for bad_path in files_black_list
                        if current_path.startswith(bad_path)
                ):
                    files.add(current_path)
                    logger.info(current_path)
        return files

    def __log_all_revisions(self):
        logger = self.__get_logger('all_commits.json')
        changes = dict()

        for entry in self.svn_log:
            changes[entry['revision']] = {
                'author': entry['author'],
                'change': entry['msg'],
                'date': str(entry['date']),
                'paths': [path['text'] for path in entry['paths']]
            }

        logger.info(json.dumps(changes))

    def __init__(self, root_path=None, svn_log=None, log_path=None):
        self.root_path = root_path
        self.svn_log = svn_log
        self.log_path = log_path
        self.__log_all_revisions()

    @property
    def has_changes(self):
        return len(self.svn_log) > 0

    def filter_by_tickets(self, statrek_queues):
        if not statrek_queues:
            return self
        ticket_pattern = '(%s)-(\d+)' % '|'.join(statrek_queues)
        logger = self.__get_logger('filter_commits_by_tickets')
        result = []
        for entry in self.svn_log:
            if entry['msg'] and re.search(ticket_pattern, entry['msg']):
                result.append(entry)
                logger.info('Add commit by ticket: ' + str(entry['revision']))
        self.svn_log = result
        return self

    def filter_blacklisted_files(self, files_black_list):
        logger = self.__get_logger('filter_commits_by_files')
        all_files_paths = self.__all_files_paths(files_black_list + ['.svn'])

        filtered_log = []
        for log_entry in self.svn_log:
            for entry_path in log_entry['paths']:
                current_path = self._remove_branch_path(entry_path['text'])
                if current_path in all_files_paths:
                    filtered_log.append(log_entry)
                    logger.info(
                        'Add commit by file: ' + str(log_entry['revision']) +
                        '. File: ' + current_path
                    )
                    break
        self.svn_log = filtered_log
        return self

    def filter_whitelisted_files(self, files_white_list):
        if not files_white_list:
            return self

        def __does_log_entry_match(log_entry):
            for entry_path in log_entry['paths']:
                current_path = self._remove_branch_path(entry_path['text'])
                if any(current_path.startswith(whitelisted_file) for whitelisted_file in files_white_list):
                    return True

            logging.info(
                'Log entry ' + str(log_entry['revision']) +
                ' is excluded as it has no whitelisted files ' + str(files_white_list)
            )

            return False

        self.svn_log = [log_entry for log_entry in self.svn_log if __does_log_entry_match(log_entry)]
        return self

    def _remove_branch_path(self, path):
        branch_name_part = re.search(self.PATH_PATTERN, path)
        if branch_name_part:
            return path[branch_name_part.end():]
        return path

    def filter_by_users(self, black_list):
        logger = self.__get_logger('filter_commits_by_user')
        result = []
        for entry in self.svn_log:
            if entry['author'] not in black_list:
                result.append(entry)
                logger.info(
                    'Add commit by user: ' + str(entry['revision']) +
                    '. User: ' + entry['author']
                )
            else:
                logger.info(
                    'Remove commit by user: ' + str(entry['revision']) +
                    '. User: ' + entry['author']
                )
        self.svn_log = result
        return self
