# coding=utf-8

import base64
import os

import shutil
from sandbox.sdk2 import Vault


class GpgKey(object):
    """
    Context manager for using gpg keys pair.

    Usage:
        with gnupg.GpgKey('<vault_secret_key_name>', '<vault_public_key_name>'):
            ...
    """
    gnupg_dir = os.path.expanduser('~/.gnupg')
    TEMPGNUPG_DIR = os.path.expanduser('~/.local/tempgnupg')

    def __init__(self, secret_key_name, public_key_name):
        """
        :param secret_key_name: vault item that contains secret gpg key in base64 representation
        :param public_key_name: vault item that contains public gpg key in base64 representation
        """
        self.__secret_key = secret_key_name
        self.__public_key = public_key_name

    def __enter__(self):
        if os.path.exists(self.TEMPGNUPG_DIR):
            shutil.rmtree(self.TEMPGNUPG_DIR)

        if os.path.exists(self.gnupg_dir):
            shutil.move(self.gnupg_dir, self.TEMPGNUPG_DIR)
        os.mkdir(self.gnupg_dir)
        Vault.data(self.__secret_key)
        with open(os.path.join(self.gnupg_dir, 'secring.gpg'), 'wb') as sec:
            sec.write(base64.b64decode(Vault.data(self.__secret_key)))

        with open(os.path.join(self.gnupg_dir, 'pubring.gpg'), 'wb') as pub:
            pub.write(base64.b64decode(Vault.data(self.__public_key)))

        return self

    def __exit__(self, exc_type, exc_val, exc_tb):
        shutil.rmtree(self.gnupg_dir)
        if os.path.exists(self.TEMPGNUPG_DIR):
            shutil.move(self.TEMPGNUPG_DIR, self.gnupg_dir)
