# coding=utf-8

from sandbox.sdk2 import svn
from sandbox.sdk2.path import Path
from sandbox.sdk2.helpers import (
    subprocess as sp,
    ProcessLog
)


class SvnHelper(object):

    COMMIT_USER = 'zomb-sandbox-rw'

    def __init__(
        self,
        module_path=None,
        task=None,
        root_path=None,
        branch_path=None,
        revision=None
    ):
        self.arcadia_root = root_path
        self.branch_path = branch_path
        self.relative_module_path = module_path
        self.task = task
        self.revision = revision if revision else None

    @property
    def __absolute_ya_path(self):
        return str(
            Path(self.arcadia_root).joinpath('arcadia').joinpath('ya')
        )

    def _get_svn_url(self, parsed_url):
        if parsed_url.trunk:
            return '/arc/trunk/arcadia'
        return self._get_branch(parsed_url)

    def _get_branch(self, parsed_url):
        if parsed_url.branch:
            return '/arc/branches/' + parsed_url.branch + '/arcadia'
        if parsed_url.tag:
            return '/arc/tags/' + parsed_url.tag + '/arcadia'
        return parsed_url.path

    @property
    def absolute_build_mk_path(self):
        return str(
            Path(self.arcadia_root).joinpath('arcadia').joinpath('market').joinpath('devtools').joinpath('build.mk')
        )

    @property
    def absolute_module_path(self):
        return str(
            Path(self.arcadia_root).joinpath(self.relative_module_path)
        )

    def checkout_branch(self):
        svn.Arcadia.checkout(
            self.branch_path,
            self.arcadia_root,
            depth=svn.Svn.Depth.FILES,
            revision=self.revision
        )
        return self

    def update_arcadia_root(self):
        svn.Arcadia.update(
            str(Path(self.arcadia_root).joinpath('arcadia')),
            depth=svn.Svn.Depth.FILES,
            revision=self.revision
        )
        return self

    def update_market(self):
        svn.Arcadia.update(
            str(Path(self.arcadia_root).joinpath('arcadia').joinpath('market')),
            parents=True,
            depth=svn.Svn.Depth.INFINITY,
            revision=self.revision
        )
        return self

    def update_arcadia_tests_data_recognize_dict(self):
        svn.Arcadia.update(
            str(
                Path(self.arcadia_root).joinpath('arcadia_tests_data').joinpath('recognize').joinpath('dict.dict')
            ),
            parents=True,
            depth=svn.Svn.Depth.INFINITY,
            revision=self.revision
        )
        return self

    def update_src(self):
        svn.Arcadia.update(
            self.absolute_module_path,
            parents=True,
            depth=svn.Svn.Depth.INFINITY,
            revision=self.revision
        )
        return self

    def update_build_mk(self):
        svn.Arcadia.update(
            self.absolute_build_mk_path,
            parents=True,
            revision=self.revision
        )
        return self

    def ya_clone(self):
        ya = svn.Arcadia.export('arcadia:/arc/trunk/arcadia/ya', 'ya')
        parsed_url = self._get_svn_url(
            parsed_url=svn.Arcadia.parse_url(url=self.branch_path)
        )

        ya_cmd = '{ya_path} -v clone --branch={branch}'.format(
            ya_path=ya,
            branch=parsed_url
        )

        if self.revision:
            ya_cmd = ya_cmd + ' --revision={revision}'.format(revision=self.revision)
        with ProcessLog(self.task, logger='ya_clone') as pl:
            sp.check_call(
                ya_cmd,
                shell=True,
                cwd=self.arcadia_root,
                stdout=pl.stdout,
                stderr=sp.STDOUT
            )
        return self

    def checkout_deps(self):
        ya_cmd = self.__absolute_ya_path + ' ' + 'make -j0 --checkout'
        with ProcessLog(self.task, logger='checkout_deps') as pl:
            sp.check_call(
                ya_cmd,
                shell=True,
                cwd=self.absolute_module_path,
                stdout=pl.stdout,
                stderr=sp.STDOUT
            )
        return self

    def checkout_market_deps(self):
        ya_cmd = self.__absolute_ya_path + ' ' + 'make -j0 --checkout'
        with ProcessLog(self.task, logger='checkout_deps_market') as pl:
            sp.check_call(
                ya_cmd,
                shell=True,
                cwd=str(
                    Path(self.arcadia_root).joinpath('arcadia').joinpath('market')
                ),
                stdout=pl.stdout,
                stderr=sp.STDOUT
            )
        return self

    def branch_start_revision(self, branch_url):
        log = svn.Arcadia.log(
            url=branch_url,
            limit=1,
            stop_on_copy=True,
            revision_from=1,
            revision_to='HEAD',
            timeout=60
        )
        return log[0]['revision']

    def branch_log(self, revision_from, revision_to='HEAD'):
        branch_log = svn.Arcadia.log(
            url=self.branch_path,
            revision_from=revision_from,
            revision_to=revision_to,
            fullpath=True,
            timeout=600
        )
        return branch_log

    def last_changelog_revision(self, branch_url):
        changelog_url = svn.Arcadia.append(
            url=svn.Arcadia.append(
                url=branch_url,
                path=self.relative_module_path
            ),
            path='debian/changelog'
        )
        log = svn.Arcadia.log(
            url=changelog_url,
            revision_from='HEAD',
            revision_to=1,
            limit=1,
            timeout=60
        )
        last_commit = log[0]['revision']
        return last_commit

    def commit_file_changelog(self):
        work_dir = str(
            Path(self.absolute_module_path).joinpath('debian').joinpath('changelog')
        )
        msg = 'commit changelog for {module_path} [no merge]'.format(
            module_path=self.relative_module_path
        )
        commit_user = self.COMMIT_USER
        svn.Arcadia.commit(work_dir, msg, commit_user)

    def commit_task_history(self, path):
        msg = 'commit task history for {module_path} [no merge]'.format(
            module_path=self.relative_module_path
        )
        status = svn.Arcadia.status(path=path)
        if status[:1] == '?':
            svn.Arcadia.add(path=path, parents=True)
        commit_user = self.COMMIT_USER
        svn.Arcadia.commit(path, msg, commit_user)

    def last_branch_revision(self, branch_url):
        log = svn.Arcadia.log(
            url=branch_url,
            revision_from='HEAD',
            revision_to=1,
            limit=1,
            timeout=60
        )
        return log[0]['revision']
