import datetime
import json
import logging
import os
import re
import shutil
from sandbox import sdk2
from sandbox.common.errors import SubprocessError
from sandbox.common.types.client import Tag
from sandbox.common.types import resource as ctr
from sandbox.projects.common.environments import SandboxJavaJdkEnvironment, PipEnvironment
from sandbox.projects.common.nanny import nanny
from sandbox.sdk2.helpers import subprocess

# 1. matcher
from sandbox.projects.market.resources import MARKET_DATA_MATCHER2
from sandbox.projects.market.resources import MARKET_DATA_MATCHER2_LAST_SESSION_ID
# 2. clutcher
from sandbox.projects.market.resources import MARKET_DATA_CLUTCHER2
from sandbox.projects.market.resources import MARKET_DATA_CLUTCHER2_LAST_SESSION_ID
# 3. skutcher
from sandbox.projects.market.resources import MARKET_DATA_SKUTCHER
from sandbox.projects.market.resources import MARKET_DATA_SKUTCHER_LAST_SESSION_ID


class IrService:
    matcher = 'matcher'
    clutcher = 'clutcher'
    skutcher = 'mbo-skubd'
    all_services = [matcher, clutcher, skutcher]  # MUST be in sync with ir-getter "--ir-service" values


class MarketIrDataGetter(nanny.ReleaseToNannyTask2, sdk2.Task):
    """
    Market IR data-getter
    """

    _MARKET_OWNER = 'MARKET'
    last_dump_session_id = 'last_dump_session_id'
    MARKET_IR_DATAGETTER_JAR = 'market-ir-datagetter.jar'

    @staticmethod
    def _get_service_data_resource(comp):
        if comp == IrService.matcher:
            return MARKET_DATA_MATCHER2.__name__
        elif comp == IrService.clutcher:
            return MARKET_DATA_CLUTCHER2.__name__
        elif comp == IrService.skutcher:
            return MARKET_DATA_SKUTCHER.__name__
        else:
            raise RuntimeError('unknown component {}'.format(comp))

    @staticmethod
    def _get_service_last_sid_resource(comp):
        if comp == IrService.matcher:
            return MARKET_DATA_MATCHER2_LAST_SESSION_ID
        elif comp == IrService.clutcher:
            return MARKET_DATA_CLUTCHER2_LAST_SESSION_ID
        elif comp == IrService.skutcher:
            return MARKET_DATA_SKUTCHER_LAST_SESSION_ID
        else:
            raise RuntimeError('unknown component {}'.format(comp))

    @staticmethod
    def _get_service_last_sid_resource_name(comp):
        if comp == IrService.matcher:
            return MARKET_DATA_MATCHER2_LAST_SESSION_ID.__name__
        elif comp == IrService.clutcher:
            return MARKET_DATA_CLUTCHER2_LAST_SESSION_ID.__name__
        elif comp == IrService.skutcher:
            return MARKET_DATA_SKUTCHER_LAST_SESSION_ID.__name__
        else:
            raise RuntimeError('unknown component {}'.format(comp))

    class Requirements(sdk2.Task.Requirements):
        environments = [SandboxJavaJdkEnvironment('11.0.2'), PipEnvironment('yandex-yt'), ]
        client_tags = Tag.SAS  # run in the same DC as hahn yt cluster

    class Parameters(sdk2.Task.Parameters):
        yt_secret_name = sdk2.parameters.String(
            'Yt token yav secret name', default='sec-01dk6zp9vxm9ffpzmvacp0adhr', required=True)
        java_opts = sdk2.parameters.String("Arguments for java", default='-Djava.net.preferIPv6Addresses=true')
        additional_args = sdk2.parameters.String(
            "Additional arguments to be appended to `java -jar`")
        ir_getter_jar_resource_id = sdk2.parameters.Integer(
            "ir-datagetter jar resource id", default=0)
        ir_getter_jar_resource_path = sdk2.parameters.String("ir-datagetter jar resource path")
        ignore_task_stop_on_same_sids = sdk2.parameters.Bool(
            "Ignore task stop when session id is not changed", default=False)
        create_nanny_release = sdk2.parameters.Bool("Create Nanny release on success", default=True)
        create_resource_bundle = sdk2.parameters.Bool("Create resource bundle on success", default=True)
        ir_service = sdk2.parameters.String("Type of generated resource",
                                            choices=[(x, x) for x in IrService.all_services],
                                            required=True)
        compress = sdk2.parameters.Bool("Do compress each file with gzip", default=False)
        released_resources_ttl_days = sdk2.parameters.Integer("TTL in days for released resources", default=7)
        with sdk2.parameters.String("Environment type", required=True) as environment_type:
            environment_type.values["stable"] = environment_type.Value("STABLE")
            environment_type.values["prestable"] = environment_type.Value("PRESTABLE")
            environment_type.values["testing"] = environment_type.Value("TESTING")

    @staticmethod
    def convert_env_type_to_yt_path_name(env_type):
        return 'production' if env_type == 'stable' else env_type

    @staticmethod
    def get_resource_path(resource_id, resource_type, resource_file_name):
        if resource_id is not None and resource_id > 0:
            return str(sdk2.ResourceData(sdk2.Resource.find(id=resource_id).first()).path)
        return os.path.join(str(sdk2.ResourceData(sdk2.Resource.find(type=resource_type, state="READY")
                                                  .order(-sdk2.Resource.id).first()).path), resource_file_name)

    def _get_last_session_id(self):
        attrs = dict(env=self.Parameters.environment_type)
        resp = sdk2.Resource.find(
            resource_type=self._get_service_last_sid_resource(self.Parameters.ir_service),
            state=ctr.State.READY,
            owner=self._MARKET_OWNER,  # https://st.yandex-team.ru/CSADMIN-24017
            attrs=attrs
        )
        if resp.count == 0:
            return None
        sid_dir = str(sdk2.ResourceData(resp.first()).path)
        with open(os.path.join(sid_dir, self.last_dump_session_id), 'r') as fin:
            return fin.readline().strip()


    def _deref_if_link(self, yt_cluster, yt_token, p):
        import yt.wrapper
        client = yt.wrapper.YtClient(yt_cluster, yt_token)
        x = client.get(p + '&/@')
        tp = 'target_path'
        return x[tp] if tp in x else p

    @staticmethod
    def _extract_single_sid_from_path(p):
        sids = re.findall(r'\d{8}_\d{4}', p)
        if len(sids) != 1:
            raise Exception('expected single session id to be extracted from {}, got {}: {}'.format(p, len(sids), sids))
        return sids[0]

    def on_execute(self):
        start_time = datetime.datetime.now().strftime("%Y-%m-%d %H:%M:%S")

        yt_secret = sdk2.yav.Secret(self.Parameters.yt_secret_name)
        yt_token = yt_secret.data()["yt.token"]
        mbo_export_path = '//home/market/{}/mbo/export/recent'.format(
            self.convert_env_type_to_yt_path_name(self.Parameters.environment_type))
        yt_cluster = 'hahn'
        ir_service = self.Parameters.ir_service
        compress = self.Parameters.compress
        dereffed_mbo_export_path = self._deref_if_link(yt_cluster, yt_token, mbo_export_path)
        current_sid = self._extract_single_sid_from_path(dereffed_mbo_export_path)

        last_sid = self._get_last_session_id()
        logging.info('current_sid={}, last_sid={}'.format(current_sid, last_sid))
        if (last_sid is not None) and (current_sid == last_sid):
            message = 'current_sid and last_sid are equal. '
            if self.Parameters.ignore_task_stop_on_same_sids:
                logging.info(message + 'force command start because ignore_task_stop_on_same_sids=True')
            else:
                logging.info(message + 'skip any further actions')
                return
        if last_sid is None:
            last_sid = ""

        cwd = os.getcwd()

        with sdk2.helpers.ProcessLog(self, logger='ir-getter') as pl:
            my_env = os.environ.copy()
            my_env["YT_TOKEN"] = yt_token

            command = 'java {java_opts} ' \
                      ' -cp {jar_path} ' \
                      ' ru.yandex.market.ir.datagetter.AppCli ' \
                      ' --mbo-export-path {mbo_export_path} ' \
                      ' --last-processed-sid "{last_processed_sid}" ' \
                      ' --yt-cluster {yt_cluster} ' \
                      ' --ir-service {ir_service} ' \
                      ' {compress} ' \
                      ' {additional_args} '.format(
                java_opts=self.Parameters.java_opts,
                jar_path=self.get_resource_path(self.Parameters.ir_getter_jar_resource_id,
                                                self.Parameters.ir_getter_jar_resource_path,
                                                self.MARKET_IR_DATAGETTER_JAR),
                mbo_export_path=dereffed_mbo_export_path,
                last_processed_sid=last_sid,
                yt_cluster=yt_cluster,
                ir_service=ir_service,
                compress=' --compress ' if compress else '',
                additional_args=self.Parameters.additional_args
            )
            logging.info("executing command: {}".format(command))
            retcode = subprocess.Popen(
                command,
                shell=True,
                stdout=pl.stdout,
                stderr=pl.stderr,
                env=my_env,
                cwd=cwd
            ).wait()
            if retcode:
                raise SubprocessError('Java process exited with non-zero return code {}'.format(retcode))

        # mbcs stands for "model-barcodes.txt" and lm for light-mathcer
        # 1. prepare current session id file
        ir_service = self.Parameters.ir_service
        last_session_id_dst = os.path.join(cwd, ir_service + '_last_session_id')
        os.makedirs(last_session_id_dst)
        with open(os.path.join(last_session_id_dst, self.last_dump_session_id), 'w') as fout:
            fout.write(current_sid)

        data_dst = os.path.join(cwd, ir_service)

        # 5. maybe create new resource
        if self.Parameters.create_resource_bundle:
            # 5.1
            resource_description = "data bundle for {}, generated {} in {} env".format(
                ir_service,
                start_time,
                self.Parameters.environment_type
            )
            sdk2.ResourceData(sdk2.Resource[self._get_service_data_resource(self.Parameters.ir_service)](
                self,
                resource_description,
                str(data_dst),
                env=self.Parameters.environment_type,
                date=start_time)).ready()

            # 5.2
            last_session_id_resource_description = "last session id {} resource for {}, generated {} in {} env".format(
                current_sid,
                ir_service,
                start_time,
                self.Parameters.environment_type
            )
            sdk2.ResourceData(sdk2.Resource[self._get_service_last_sid_resource_name(self.Parameters.ir_service)](
                self,
                last_session_id_resource_description,
                str(last_session_id_dst),
                env=self.Parameters.environment_type,
                date=start_time)).ready()

    def on_success(self, prev_status):
        sdk2.Task.on_success(self, prev_status)

        # https://st.yandex-team.ru/CSADMIN-24017
        if self.owner != self._MARKET_OWNER:
            logging.warning("Owner is %s. Skip any further action", self.owner)
            return

        if self.Parameters.create_nanny_release:
            nanny.ReleaseToNannyTask2.on_release(self, dict(
                release_comments=None,
                release_status=self.Parameters.environment_type,
                release_subject="Auto release from sandbox ir {} data-getter".format(self.Parameters.ir_service),
                releaser=self.author
            ))

    def mark_released_resources(self, status, ttl="inf"):
        ttl = self.Parameters.released_resources_ttl_days
        return sdk2.Task.mark_released_resources(self, status, ttl)

    @staticmethod
    def _concat_filtered(d1, d2, d2_key_pred):
        d2_filtered = {k: v for k, v in d2.iteritems() if d2_key_pred(k)}
        return dict(d1.items() + d2_filtered.items())

