import datetime
import logging
import os

from sandbox import sdk2
from sandbox.common.errors import SubprocessError
from sandbox.common.types.client import Tag
from sandbox.projects.common.environments import SandboxJavaJdkEnvironment, PipEnvironment
from sandbox.projects.common.nanny import nanny
from sandbox.projects.market.resources import MARKET_DATA_SMART_MATCHER_SNAPSHOT
from sandbox.sdk2.helpers import subprocess


class MarketIrSmartmatcherSnapshot(nanny.ReleaseToNannyTask2, sdk2.Task):
    """
    Market IR data-getter
    """

    _MARKET_OWNER = 'MARKET'
    MARKET_IR_DATAGETTER_JAR = 'snapshot-creator.jar'




    class Requirements(sdk2.Task.Requirements):
        cores = 10
        ram = 8192
        disk_space = 100000
        environments = [SandboxJavaJdkEnvironment('11.0.2'), PipEnvironment('yandex-yt'), ]
        client_tags = Tag.SAS  # run in the same DC as hahn yt cluster

    class Parameters(sdk2.Task.Parameters):
        yt_secret_name = sdk2.parameters.String(
            'Yt token yav secret name', default='sec-01dk6zp9vxm9ffpzmvacp0adhr', required=True)
        java_opts = sdk2.parameters.String("Arguments for java", default='-Djava.net.preferIPv6Addresses=true')
        additional_args = sdk2.parameters.String(
            "Additional arguments to be appended to `java -jar`")
        snapshot_creator_resource_id = sdk2.parameters.Integer(
            "ir-datagetter jar resource id", default=0)
        ir_getter_jar_resource_path = sdk2.parameters.String("ir-datagetter jar resource path")
        create_nanny_release = sdk2.parameters.Bool("Create Nanny release on success", default=True)
        create_resource_bundle = sdk2.parameters.Bool("Create resource bundle on success", default=True)
        ir_service = MARKET_DATA_SMART_MATCHER_SNAPSHOT.__name__
        released_resources_ttl_days = sdk2.parameters.Integer("TTL in days for released resources", default=7)
        threads = sdk2.parameters.Integer("Threads number", default=10)
        categories = sdk2.parameters.String("Categories")
        snapshot_yt_path = sdk2.parameters.String("YT Smartmatcher Snapshot path", default="//home/market/production/ir/smartmatcher/snapshot/recent")

        with sdk2.parameters.String("Environment type", required=True) as environment_type:
            environment_type.values["stable"] = environment_type.Value("STABLE")
            environment_type.values["prestable"] = environment_type.Value("PRESTABLE")
            environment_type.values["testing"] = environment_type.Value("TESTING")

    @staticmethod
    def convert_env_type_to_yt_path_name(env_type):
        return 'production' if env_type == 'stable' else env_type

    @staticmethod
    def get_resource_path(resource_id, resource_type, resource_file_name):
        if resource_id is not None and resource_id > 0:
            return str(sdk2.ResourceData(sdk2.Resource.find(id=resource_id).first()).path)
        return os.path.join(str(sdk2.ResourceData(sdk2.Resource.find(type=resource_type, state="READY")
                                                  .order(-sdk2.Resource.id).first()).path), resource_file_name)



    def on_execute(self):
        start_time = datetime.datetime.now().strftime("%Y-%m-%d %H:%M:%S")

        yt_secret = sdk2.yav.Secret(self.Parameters.yt_secret_name)
        yt_token = yt_secret.data()["yt.token"]
        yt_cluster = 'hahn'
        ir_service = self.Parameters.ir_service

        cwd = os.getcwd()

        with sdk2.helpers.ProcessLog(self, logger='sm-snapshot-getter') as pl:
            my_env = os.environ.copy()
            my_env["YT_TOKEN"] = yt_token

            command = 'java {java_opts} ' \
                      ' -cp {jar_path} ' \
                      ' ru.yandex.market.ir.Main ' \
                      ' --snapshot-yt-path {snapshot_path} ' \
                      ' --threads "{threads}" ' \
                      ' --yt-cluster {yt_cluster} ' \
                      ' --only-msku ' \
                      ' --zip-files'  \
                      ' --categories ${categories} ' \
                      ' {additional_args} '.format(
                java_opts=self.Parameters.java_opts,
                jar_path=self.get_resource_path(self.Parameters.ir_getter_jar_resource_id,
                                                self.Parameters.ir_getter_jar_resource_path,
                                                self.MARKET_IR_DATAGETTER_JAR),
                snapshot_path=self.Parameters.snapshot_yt_path,
                threads=self.Parameters.threads,
                yt_cluster=yt_cluster,
                categories=self.Parameters.categories if self.Parameters.categories else '',
                additional_args=self.Parameters.additional_args
            )
            logging.info("executing command: {}".format(command))
            retcode = subprocess.Popen(
                command,
                shell=True,
                stdout=pl.stdout,
                stderr=pl.stderr,
                env=my_env,
                cwd=cwd
            ).wait()
            if retcode:
                raise SubprocessError('Java process exited with non-zero return code {}'.format(retcode))


        data_dst = os.path.join(cwd, 'smartmatcher-data')

        # 5. maybe create new resource
        if self.Parameters.create_resource_bundle:
            # 5.1
            resource_description = "data bundle for {}, generated {} in {} env".format(
                ir_service,
                start_time,
                self.Parameters.environment_type
            )
            sdk2.ResourceData(sdk2.Resource[MARKET_DATA_SMART_MATCHER_SNAPSHOT.__name__](
                self,
                resource_description,
                str(data_dst),
                env=self.Parameters.environment_type,
                date=start_time)).ready()



    def on_success(self, prev_status):
        sdk2.Task.on_success(self, prev_status)

        # https://st.yandex-team.ru/CSADMIN-24017
        if self.owner != self._MARKET_OWNER:
            logging.warning("Owner is %s. Skip any further action", self.owner)
            return

        if self.Parameters.create_nanny_release:
            nanny.ReleaseToNannyTask2.on_release(self, dict(
                release_comments=None,
                release_status=self.Parameters.environment_type,
                release_subject="Auto release from sandbox ir {} data-getter".format(self.Parameters.ir_service),
                releaser=self.author
            ))

    def mark_released_resources(self, status, ttl="inf"):
        ttl = self.Parameters.released_resources_ttl_days
        return sdk2.Task.mark_released_resources(self, status, ttl)

    @staticmethod
    def _concat_filtered(d1, d2, d2_key_pred):
        d2_filtered = {k: v for k, v in d2.iteritems() if d2_key_pred(k)}
        return dict(d1.items() + d2_filtered.items())




if __name__ == '__main__':
    task = MarketIrSmartmatcherSnapshot()
    print(task)
