# -*- coding: utf-8 -*-

from os import path

from sandbox.projects import resource_types
from sandbox.projects.common import constants as consts
from sandbox.sandboxsdk import task
from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk.paths import make_folder
from sandbox.sandboxsdk.process import run_process

import resources


class BuildMarketGuruAssistantData(task.SandboxTask):
    type = 'BUILD_MARKET_GURU_ASSISTANT_DATA'

    input_parameters = [
    ]

    execution_space = 10 * 1024  # 10 Gb

    def get_assistant_base_builder(self):
        from sandbox.sandboxsdk import svn

        svn_url = svn.Arcadia.append(self.ctx[consts.ARCADIA_URL_KEY], 'market/assistant/assistantbase/')

        path_to = make_folder('assistantbase')
        svn.Arcadia.export(svn_url, path_to)

        self.ctx['cfg_programm_assistantbase'] = path_to

        return path_to

    def get_data(self):
        path_to = make_folder('data')
        path_result = make_folder('data/result')
        path_gurudata = make_folder('data/gurudata')

        prev_data_arch = resources.get_newest_resource(self, resource_types.MARKET_GURU_ASSISTANT_DATA,
                                                       'previous guruassistant data')
        auto_data_arch = resources.get_newest_resource(self, resource_types.MARKET_GURU_ASSISTANT_AUTO_WHEELS,
                                                       'auto.ru data')
        guru_data_arch = resources.get_newest_resource(self, resource_types.MARKET_MBO_MODEL_PARAMETERS,
                                                       'mbo category data') + '/mbo_stuff/stable/models/parameters_90490.pb'

        cmd = 'tar --transform="s/guruassistant/old_data/" -C {}  -xf {}'.format(path_to, prev_data_arch)
        run_process(cmd, check=True, wait=True, log_prefix='get_data')

        auto_data_file = path.join(path_to, 'auto_wheels.xml')
        with open(auto_data_file, 'wb') as out:
            cmd = 'gzip -d -c {}'.format(auto_data_arch)
            run_process(cmd, check=True, wait=True, log_prefix='get_data', outputs_to_one_file=False, stdout=out)

        cmd = 'cp {} {}'.format(guru_data_arch, path_gurudata)
        run_process(cmd, check=True, wait=True, log_prefix='get_data')

        self.ctx['cfg_data_all'] = path_to
        self.ctx['cfg_data_auto'] = auto_data_file
        self.ctx['cfg_data_guru'] = path_gurudata
        self.ctx['cfg_data_old'] = path.join(path_to, 'old_data')
        self.ctx['cfg_result_path'] = path_result

        return path_to

    def generate_config(self):
        config = [
            "[programs]\n",
            "root: " + self.ctx['cfg_programm_assistantbase'] + "\n",
            "assistant: " + self.ctx['cfg_programm_printguruass'] + "\n",
            "mboparamdumper: " + self.ctx['cfg_programm_paramsdump'] + "\n",
            "\n",

            "[data]\n",
            "mbodatadir: " + self.ctx['cfg_data_guru'] + "\n",
            "autobase:" + self.ctx['cfg_data_auto'] + "\n",
            "\n",

            "[output]\n",
            "dir_output: " + self.ctx['cfg_result_path'] + "\n",
            "create_subdir: False\n",
            "upload_to_sandbox: False\n",
            "\n",

            "[report]\n",
            "mail_to: market-kgb-guruassistant@yandex-team.ru\n",
            "mail_from: market-kgb-guruassistant@yandex-team.ru\n",
            "\n",
            "send_mail: True\n",
            "print_to_stdout: True\n",
            "\n",

            "[test]\n",
            "dir_old_result: " + self.ctx['cfg_data_old'] + "/\n",
            "test_requests: " + path.join(self.ctx['cfg_programm_assistantbase'], "test/tires_test_req.txt") + "\n",
            "min_result_size: 100 * 1024\n",
            "allow_disappear: False\n",
            "allow_uniterror: False\n",
            "\n",

            "testing: False\n"
        ]

        config_file_path = path.join(self.ctx['cfg_data_all'], "config_test.ini")
        with open(config_file_path, "w") as config_file:
            config_file.writelines(config)

        self.ctx['cfg_file'] = config_file_path

        return config_file_path

    def create_resources(self, result_file_name):
        assistantbase_resource = self.create_resource(
            description='assistantbase (for test purpose)',
            resource_path=self.ctx['cfg_programm_assistantbase'],
            resource_type=resource_types.OTHER_RESOURCE,
            attributes={
                'ttl': 4
            }
        )

        data_resource = self.create_resource(
            description='assistantbase data',
            resource_path=self.ctx['cfg_data_all'],
            resource_type=resource_types.OTHER_RESOURCE,
            attributes={
                'ttl': 4
            }
        )

        self.mark_resource_ready(assistantbase_resource.id)
        self.mark_resource_ready(data_resource.id)

        result_resource = resources.copy_and_release(
            task=self,
            filename=result_file_name,
            description='guruassistant data',
            resource_type=resource_types.MARKET_GURU_ASSISTANT_DATA,
            attributes={
                'type': 'sandbox',
                'ttl': 14
            }
        )

        return result_resource

    def on_execute(self):
        from sys import executable as python
        from build_programs import build_binaries
        import report

        resources.get_last_release(self)
        ass_path = self.get_assistant_base_builder()

        if not resources.get_bin_by_type(self, resource_types.MARKET_GURU_ASSISTANT_PRINTGURUASS,
                                         'cfg_programm_printguruass') \
                or not resources.get_bin_by_type(self, resource_types.MARKET_MBO_PARAMSDUMP, 'cfg_programm_paramsdump'):
            build_binaries(self)

        self.get_data()
        ass_cfg = self.generate_config()

        cmd = '{} {} {}'.format(python, path.join(ass_path, 'assistantbase.py'), ass_cfg)
        run_process(cmd, check=True, wait=True, log_prefix='assistantbase')

        result = self.create_resources(path.join(self.ctx['cfg_result_path'], 'assistantbase.tar.gz'))
        report.report(self, ['market-kgb-guruassistant@yandex-team.ru'], result)

    def on_release(self, additional_parameters):
        for res in self.list_resources():
            if channel.sandbox.get_resource_attribute(res, 'ttl'):
                channel.sandbox.set_resource_attribute(res, 'ttl', '14')


__Task__ = BuildMarketGuruAssistantData
