from __future__ import print_function

from os import path
import datetime


def report(task, mail_to, result):
    from sandbox.sandboxsdk.channel import channel
    rep = Report(task, result)

    channel.sandbox.send_email(mail_to, None, rep.subject, rep.message)


class Report(object):
    ERROR_NONE = 0
    ERROR_WARNING = 1
    ERROR_ERROR = 2
    ERROR_CRITICAL = 4

    def __init__(self, task, result):
        self.message = ""
        self.subject = " Guruassistant base builder report"
        self.__task = task
        self.__errors_level = Report.ERROR_NONE
        self.__type = "SANDBOX"

        self.__report_from_files()

        self.message += "Result: %s\n" % result.proxy_url

        if self.__errors_level & Report.ERROR_CRITICAL:
            self.subject = "[CRITICAL]" + self.subject
        elif self.__errors_level & Report.ERROR_ERROR:
            self.subject = "[ERROR]" + self.subject
        elif self.__errors_level & Report.ERROR_WARNING:
            self.subject = "[WARNING]" + self.subject
        else:
            self.message = "All is ok\n" + self.message

        self.subject = "[" + self.__type + "]" + self.subject

        t = datetime.datetime.now()
        self.message += ("\nReport generated at: " + t.strftime("%Y-%m-%d %H:%M:%S") + "\n")

    def __report_from_files(self):
        (err_files, except_file) = self.__error_files()
        if except_file:
            self.message = "\nException occured while generating data:\n%s    \n" % except_file.proxy_url

        if err_files:
            if len(err_files) == 1:
                self.message = "\nThere is 1 non empty error log:\n"
            else:
                self.message = "\nThere are %d non empty error logs\n" % len(err_files)
            for r in err_files:
                self.message += "    %s: %s\n" % (r.file_name, r.proxy_url)

        self.message += "\nResult size: %d\n" % self.__calcresult_size()

    def __calcresult_size(self):
        import glob
        res = 0
        for fn in glob.glob(path.join(self.__task.ctx['cfg_result_path'], "assistantbase/*")):
            res += path.getsize(fn)
        return res

    def __error_files(self):
        import glob
        from projects import resource_types
        import resources
        ret = []
        for fn in glob.glob(path.join(self.__task.ctx['cfg_result_path'], "error*")):
            if path.isfile(fn) and path.getsize(fn) > 0:
                self.__errors_level |= Report.ERROR_ERROR
                r = resources.copy_and_release(
                    task=self.__task,
                    filename=fn,
                    description='guruassistant data errors',
                    resource_type=resource_types.OTHER_RESOURCE,
                    attributes={
                        'ttl': 14
                    }
                )
                ret.append(r)

        exception_r = None
        exception_file = path.join(self.__task.ctx['cfg_result_path'], "exceptions.txt")
        if path.isfile(exception_file) and path.getsize(exception_file) > 0:
            self.__errors_level |= Report.ERROR_CRITICAL
            exception_r = resources.copy_and_release(
                task=self.__task,
                filename=exception_file,
                description='guruassistant data build errors',
                resource_type=resource_types.OTHER_RESOURCE,
                attributes={
                    'ttl': 14
                }
            )

        return (ret, exception_r)
