# coding: utf-8

import os
import socket
import time

from sandbox.projects import resource_types
from sandbox.projects.common import apihelpers
from sandbox.projects.common.search import performance as search_performance
from sandbox.projects.common.dolbilka import DolbilkaPlanner
from sandbox.projects.common.utils import wait_searcher_start

import sandbox.common.types.client as ctc
from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk.process import run_process
from sandbox.sandboxsdk.parameters import SandboxStringParameter, ResourceSelector


class CatalogerBin(ResourceSelector):
    name = 'cataloger_bin'
    description = 'Cataloger binary. Empty to build from trunk'
    resource_type = resource_types.MARKET_CATALOGER_BIN_ONLINE
    required = False


class CatalogerData(ResourceSelector):
    name = 'cataloger_data'
    description = 'Cataloger data. Empty to auto-find'
    resource_type = resource_types.MARKET_CATALOGER_DATA
    required = False


class CatalogerRequests(ResourceSelector):
    name = 'cataloger_requests'
    description = 'Cataloger requests'
    resource_type = resource_types.PLAIN_TEXT_QUERIES
    default_value = 184057567  # grepped from logs
    required = True


class GraphiteHost(SandboxStringParameter):
    name = 'graphite_host'
    description = 'Graphite host to push data. Empty to ignore'
    default_value = 'graphite-testing.market.yandex.net'


class DaemonRunner(object):
    def __init__(self, bin_path, config_path, port, daemon_name, start_timeout=600):
        self.bin_path = bin_path
        self.config_path = config_path
        self.process = None

        self.host = 'localhost'
        self.port = port
        self.daemon_name = daemon_name
        self.start_timeout = start_timeout

    def run(self):
        os.symlink(self.bin_path, 'daemon_executable')
        os.symlink(self.config_path, 'servant.cfg')
        self.process = run_process(['./daemon_executable'], wait=False, log_prefix=self.daemon_name)
        wait_searcher_start('localhost', self.port, subproc_list=[self.process], timeout=self.start_timeout)

    def stop(self):
        if self.process:
            self.process.kill()
            self.process = None

    def __enter__(self):
        self.run()
        return self

    def __exit__(self, exc_type, exc_val, exc_tb):
        self.stop()


class MarketCatalogerShooter2(search_performance.OldShootingTask, SandboxTask):
    """
        Таск приемки каталогера на производительность
    """
    type = 'MARKET_CATALOGER_SHOOTER2'
    client_tags = ctc.Tag.LINUX_PRECISE & ctc.Tag.INTEL_E5_2650

    input_parameters = (
        CatalogerBin,
        CatalogerData,
        CatalogerRequests,
        GraphiteHost
    ) + search_performance.OldShootingTask.shoot_input_parameters

    current_daemon = None
    port = 8893  # random choice

    binary_type = resource_types.MARKET_CATALOGER_BIN_ONLINE
    binary_ctx_name = CatalogerBin.name
    data_type = resource_types.MARKET_CATALOGER_DATA
    plan_type = resource_types.MARKET_CATALOGER_PLAN
    data_ctx_name = CatalogerData.name
    binary_build_task_type = 'BUILD_MARKET_CATALOGER_BIN'
    daemon_name = 'cataloger'
    reqs_ctx_name = CatalogerRequests.name

    # stats from tank.DolbiloPlugin
    new_stats_types = (
        ("latency_0.5", "Latency, q50 (usec)", "{:0.2f}"),
        ("latency_0.95", "Latency, q95 (usec)", "{:0.2f}"),
        ("latency_0.99", "Latency, q99 (usec)", "{:0.2f}"),
        ("resp_size_quantile_0.5", "Response size, q50", "{:0.0f}"),
        ("resp_size_quantile_0.95", "Response size, q95", "{:0.0f}"),
        ("max_size", "Max size", "{:0.0f}"),
    )

    @staticmethod
    def generate_config(data_path, port):
        tpl = """
CATALOG_FILENAME        {data}/catalog.xml
NAVIGATION_INFO         {data}/navigation_info.xml
MOBILE_CATALOG_FILENAME {data}/mobile_catalog.xml
REGION_OFFERS_FILENAME  {data}/category_region_offers.csv
REGION_STATS_FILENAME   {data}/category_region_stats.csv
REGION_TREE_FILENAME    {data}/geo.c2p
SHOPSTAT_FILENAME       {data}/shopstat.xml
TOP_CATEGORIES_FILENAME {data}/top_categories.xml
PAGES_FILENAME          {data}/visual.categories.templates.xml
ATTRIBUTES_FILENAME     {data}/categories-attributes.xml
GLOBAL_VENDORS_FILENAME {data}/global.vendors.xml
POPULAR_VENDORS_FILENAME {data}/popular-vendors.xml
COOKBOOKS_FILENAME      {data}/guru.cookbooks.xml
RECIPES_FILENAME      {data}/recipes.xml
VENDOR_CATEGORY_STATS_FILENAME  {data}/vendor_category_stats.pbuf.sn
CATEGORY_RESTRICTIONS   {data}/category-restrictions.pb
SHOW_LOG_PATH marketcataloger_shows.log
TSKV_OUT_PATH marketcataloger-trace.log
OFFERS_FILENAME         {data}/data.xml
TOP_CATEGORIES_PDA_FILENAME {data}/top_categories_pda.xml

HTTP_PORT                        {port}
LOG                              daemon.log
PIDFILE_DIR                      .
"""
        return tpl.format(data=data_path, port=port)

    def get_short_task_result(self):
        if not self.is_completed():
            return None
        return int(self.ctx['max_rps'])

    def push_rps(self, rps):
        if not self.ctx.get(GraphiteHost.name):
            return
        s = socket.socket(socket.AF_INET6, socket.SOCK_STREAM)
        s.connect((self.ctx[GraphiteHost.name], 2024))
        s.sendall('one_hour.%s.shootings_http.rps %s %s' % (self.daemon_name, rps, int(time.time())))
        s.close()

    def make_plan(self):
        plan_path = self.abs_path('plan')
        path_to_planner = DolbilkaPlanner.get_planner_path()
        requests = self.sync_resource(self.ctx[self.reqs_ctx_name])
        command = ' '.join([path_to_planner, '-l', requests, '-o', plan_path, '-t plain',
                            '-h', self.current_daemon.host, '-p', str(self.current_daemon.port)])

        run_process(command, log_prefix='planner')
        resource = self.create_resource('plan', plan_path, self.plan_type)
        self.mark_resource_ready(resource)
        return resource.id

    def build_binary(self):
        ctx_name = 'binary_build_task'
        if ctx_name not in self.ctx:
            self.ctx[ctx_name] = self.create_subtask(
                task_type=self.binary_build_task_type,
                description='build %s from trunk' % self.daemon_name
            ).id
            self.wait_task_completed(self.ctx[ctx_name])
        else:
            self.ctx[self.binary_ctx_name] = apihelpers.get_task_resource_id(
                task_id=self.ctx[ctx_name],
                resource_type=self.binary_type,
            )

    def on_execute(self):
        if not self.ctx.get(self.binary_ctx_name):
            self.build_binary()

        bin_path = self.sync_resource(self.ctx[self.binary_ctx_name])

        if not self.ctx[self.data_ctx_name]:
            self.ctx[self.data_ctx_name] = apihelpers.get_last_released_resource(self.data_type).id
        data_path = self.sync_resource(self.ctx[self.data_ctx_name])

        config_path = '%s.cfg' % self.daemon_name
        with open(config_path, 'w') as config:
            config.write(self.generate_config(data_path, self.port))

        with DaemonRunner(bin_path, config_path, self.port, self.daemon_name) as daemon:
            self.current_daemon = daemon
            self.make_plan()
            self._init_virtualenv()
            self._old_shoot(daemon, self.make_plan())

            # store main log as a resource
            resource = self.create_resource('daemon.log', 'daemon.log', resource_types.OTHER_RESOURCE)
            self.mark_resource_ready(resource)

        if self.ctx[GraphiteHost.name]:
            self.push_rps(int(self.ctx['max_rps']))


__Task__ = MarketCatalogerShooter2
