import logging
import os
import json

from sandbox import sdk2
from sandbox.sdk2.vcs import svn
from sandbox.sdk2.helpers import subprocess as sp


class ResultResource(sdk2.Resource):
    """result"""

    result = sdk2.parameters.String('result')


class MarketKombatReviewFinder(sdk2.Task):
    """ find revisions which affects to given project """

    class Parameters(sdk2.Task.Parameters):
        # custom parameters
        revision_from = sdk2.parameters.String('revision_from', required=True)
        revision_to = sdk2.parameters.String('revision_to', required=True)
        project_dir = sdk2.parameters.String('project_dir', required=True)

    def build_source_dir_set(self, project_dir, arcadia_url):
        SRC_ROOT_STR = '$(SOURCE_ROOT)'

        arcadia_dir = svn.Arcadia.get_arcadia_src_dir(arcadia_url)
        build_graph_json = sp.check_output(['./ya', 'make', '--checkout', '-q', '-j0', '-G', '-C', project_dir],
                                           cwd=arcadia_dir)
        build_graph = json.loads(build_graph_json)
        result_set = set()
        for build_graph_node in build_graph['graph']:
            for input_path in build_graph_node['inputs']:
                if input_path.startswith(SRC_ROOT_STR):
                    result_set.add(os.path.dirname(input_path[len(SRC_ROOT_STR):]))
        return result_set

    def find_affects_revisions(self, log_info, source_dir_set):
        TRUNK_ARCADIA = '/trunk/arcadia'
        result_revisions = []
        for log_entry in log_info:
            paths = log_entry.get('paths')
            for _, path in paths:
                if path.startswith(TRUNK_ARCADIA):
                    path = path[len(TRUNK_ARCADIA):]
                path = os.path.dirname(path)
                if path in source_dir_set:
                    revision = log_entry.get('revision')
                    logging.info('add revision: {}, cause: {}'.format(revision, path))
                    result_revisions.append(revision)
                    break
        return result_revisions

    def on_execute(self):
        project_dir = self.Parameters.project_dir
        revision_from = self.Parameters.revision_from
        revision_to = self.Parameters.revision_to

        ARCADIA_URL = svn.Arcadia.ARCADIA_TRUNK_URL

        source_dir_set = self.build_source_dir_set(project_dir, ARCADIA_URL)
        log_info = svn.Arcadia.log(ARCADIA_URL, revision_from=revision_from, revision_to=revision_to)
        result_revisions = self.find_affects_revisions(log_info, source_dir_set)

        resource = ResultResource(self, 'Output file', 'output_dir')
        resource.result = '\n'.join(map(str, result_revisions))
        data = sdk2.ResourceData(resource)
        data.path.mkdir(0o755, parents=True, exist_ok=True)
        data.path.joinpath('result.txt').write_text(resource.result)

        data.ready()
