# -*- coding: utf-8 -*-

import logging
import requests

from collections import namedtuple
from sandbox import sdk2
from sandbox.sandboxsdk import environments


class MarketKombatStatAggregator(sdk2.Task):
    class Requirements(sdk2.Task.Requirements):
        environments = [
            environments.PipEnvironment('psycopg2-binary')
        ]

    class Parameters(sdk2.Task.Parameters):
        with sdk2.parameters.Group('PostgreSQL DB parameters') as postgres_params:
            postgres_user = sdk2.parameters.String('User', default='market_kombat_junk', required=True)
            postgres_password_vault_name = sdk2.parameters.String('Password vault name', required=True)
            postgres_host = sdk2.parameters.String(
                'List of hosts',
                default='man-4yqzrwath8v72z1g.db.yandex.net,'
                        'sas-7m27xtbx4fua7jbf.db.yandex.net,'
                        'vla-y0b6un3ygbyoqrhl.db.yandex.net',
                required=True
            )
            postgres_port = sdk2.parameters.String('Port', default=6432, required=True)
            postgres_dbname = sdk2.parameters.String('DB name', default='market_kombat_junk', required=True)
        with sdk2.parameters.Group('Kombat parameters') as kombat_params:
            kombat_host = sdk2.parameters.String('Kombat host',
                                                 default='http://kombat.vs.market.yandex.net', required=True)
            kombat_battles_to_list = sdk2.parameters.Integer('Number of battles to list', default=100, required=True)

    KombatBattle = namedtuple('Battle', ['id'])
    ReportTestResult = namedtuple('TestResult', ['id', 'type', 'version', 'has_errors', 'has_degradation',
                                                 'ks_result', 'is_retry'])

    class KombatClient(object):
        def __init__(self, host):
            self.__host = host
            self.__session = None

        def _start_session(self):
            if self.__session is None:
                self.__session = requests.Session()
                self.__session.headers.update({'Content-Type': 'application/json'})
            return self.__session

        def list_battles(self, count):
            session = self._start_session()
            url = '{host}/list_battle?status=complete&sort-created-desc=1&sort-priority=0&owner=tsum&count={count}'\
                .format(
                    host=self.__host,
                    count=count
                )
            logging.info('doing request: {}'.format(url))
            return [MarketKombatStatAggregator.KombatBattle(id=int(entry['id'])) for entry in session.get(url).json()]

        def get_test_report_result(self, battle_id):
            session = self._start_session()
            url = '{host}/test_report_result?id={id}'.format(
                host=self.__host,
                id=battle_id
            )
            logging.info('doing request: {}'.format(url))
            response = session.get(url).json()
            return MarketKombatStatAggregator.ReportTestResult(
                id=battle_id,
                type=response['fire_type'],
                version=response['test_version'],
                has_errors=len(response['error']) > 0,
                has_degradation=response['time'] > 0,
                ks_result=response['kolmogorov_smirnov_result'].strip(),
                is_retry=response['is_retry'] == 1
            )

    class DBClient(object):
        def __init__(self, connection_string):
            self.__connection_string = connection_string

        def _do_fetchable_query(self, query):
            import psycopg2
            import psycopg2.extras
            with psycopg2.connect(self.__connection_string) as db_connection:
                with db_connection.cursor(cursor_factory=psycopg2.extras.DictCursor) as curs:
                    curs.execute(query)
                    return curs.fetchall()

        def _do_silent_query(self, query):
            import psycopg2
            import psycopg2.extras
            with psycopg2.connect(self.__connection_string) as db_connection:
                with db_connection.cursor() as curs:
                    curs.execute(query)

        def extract_last_battle(self):
            query = 'select last_battle_id from kombat_stat_job_state.battle_history order by run_id desc limit 1'
            result = self._do_fetchable_query(query)
            if not result:
                raise RuntimeError('Cannot extract last battle: result is empty')
            return MarketKombatStatAggregator.KombatBattle(id=int([row['last_battle_id'] for row in result][0]))

        def upload_report_results(self, results):
            logging.info('processing results before upload')
            max_battle_id = -1
            values = []
            for result in results:
                if result.has_errors or not result.version:
                    continue
                max_battle_id = max(max_battle_id, result.id)
                ks_accept = result.ks_result == 'accept'
                if result.ks_result == 'no_data':
                    ks_accept = 'null'
                values.append('(\'{version}\', \'{type}\', {battle_id}, {has_degradation}, {ks_accept}, {is_retry})'\
                    .format(
                        version=result.version,
                        type=result.type,
                        battle_id=result.id,
                        has_degradation=result.has_degradation,
                        ks_accept=ks_accept,
                        is_retry=result.is_retry
                    ))
            if len(values) == 0:
                logging.info('no need to upload')
                return
            query = 'insert into kombat_stat_job_state.releases values {};\n'.format(',\n'.join(values))
            query += 'insert into kombat_stat_job_state.battle_history (last_battle_id) values ({});'.format(
                max_battle_id)
            logging.info('uploading results using SQL query:\n' + query)
            self._do_silent_query(query)

    def _connect_to_db(self):
        if self._db is None:
            conn_str = "dbname={dbname} user={user} port={port} host={host} sslmode=require password={password}"\
                .format(
                    dbname=self.Parameters.postgres_dbname,
                    user=self.Parameters.postgres_user,
                    port=self.Parameters.postgres_port,
                    host=self.Parameters.postgres_host,
                    password=sdk2.Vault.data(self.Parameters.postgres_password_vault_name)
                )
            self._db = self.DBClient(conn_str)
        return self._db

    def _connect_to_kombat(self):
        if self._kombat is None:
            self._kombat = self.KombatClient(self.Parameters.kombat_host)
        return self._kombat

    def _collect_report_results(self, db):
        logging.info('collecting report results')
        kombat = self._connect_to_kombat()
        last_battle = db.extract_last_battle()
        recent_battles = kombat.list_battles(self.Parameters.kombat_battles_to_list)
        results = []
        logging.info('last battle id: {}'.format(last_battle.id))
        for battle in recent_battles:
            if battle.id <= last_battle.id:
                logging.info('skip battle with id: {}'.format(battle.id))
                continue
            try:
                logging.info('accept battle with id: {}'.format(battle.id))
                results.append(kombat.get_test_report_result(battle.id))
            except Exception as exception:
                logging.exception('Exception during get_test_report_result where battle.id={}: {}'
                                  .format(battle.id, exception))
        return results

    def _initialize(self):
        self._db = None
        self._kombat = None

    def on_execute(self):
        self._initialize()
        db = self._connect_to_db()
        report_results = self._collect_report_results(db)
        db.upload_report_results(report_results)
