# -*- coding: utf-8 -*-

import json
import os
import logging

from sandbox import sdk2
from sandbox.common.types import misc as ctm
from sandbox.common.utils import get_task_link
from sandbox.projects.market.devexp.front.helpers.sandbox_helpers import rich_check_call
from sandbox.projects.market.devexp.front.helpers.node import create_node_selector
from sandbox.projects.market.devexp.front.helpers.sandbox_helpers import get_resource_http_proxy_link as get_resource_link
from sandbox.projects.market.devexp.front.helpers.ubuntu import create_ubuntu_selector, setup_container
from sandbox.projects.market.devexp.front.helpers.SetEnv import SetEnv
from sandbox.projects.market.devexp.front.helpers.startrack import ReleaseTicket
from sandbox.projects.market.resources import MARKET_AUTOTEST_REPORT
from sandbox.sandboxsdk.environments import PipEnvironment
from sandbox.projects.common.arcadia import sdk as arcadia_sdk

DISK_SPACE = 3 * 1024  # 3 Gb
ST_OAUTH_TOKEN_VAULT_KEY = 'robot-crm-tifa-token'
MAIN_PASSWORD_TOKEN_VAULT_KEY = 'robot-loki-odinson-password'
ADDITIONAL_PASSWORD_TOKEN_VAULT_KEY = 'robot-helaodinsdot-password'

def _hidden_id(value):
    return '<# <div id="{id}"/> #>'.format(id=value)

def _build_test_statistic(tool, data, report_url, task_url):
    hidden_id_value = '{}-report'.format(tool)
    caption = 'Статистика прогона {}-тестов:'.format(tool)
    footer = '\n- (({} Отчет))\n- (({} Таск))\n{}'.format(report_url, task_url, _hidden_id(hidden_id_value))
    table = '{}#|{}{}|#{}'
    head = '**||Browser|Status|Tests|Passed|Failed|Broken|Skipped|Retries|Duration||**'
    row = '||{browser}|{status}|{tests}|{passed}|{failed}|{broken}|{skipped}|{retries}|{duration}||'
    body = ''
    cell = '!!({}){}!!'

    for run in data:
        duration = divmod(run['duration'] // 1000, 60)
        [hours, minutes] = divmod(duration[0], 60)
        seconds = duration[1]

        body += row.format(
            browser=run['browser'],
            status=cell.format('red' if run['status'] == 'failed' else 'green', run['status']),
            tests=cell.format('blue', run['tests']),
            passed=cell.format('green', run['passed']),
            failed=cell.format('red', run['failed']),
            broken=cell.format('yellow', run.get('broken', '-')),
            skipped=cell.format('gray', run['skipped']),
            retries=cell.format('yellow', run['retries']),
            duration='{h}:{m}:{s}'.format(h=hours, m=minutes, s=seconds)
        )

    return table.format(caption, head, body, footer)

class LilucrmOperatorWindowAutotestsHermione(sdk2.Task):
    """
    Hermione автотесты Единого Окна
    """

    class Requirements(sdk2.Task.Requirements):
        dns = ctm.DnsType.DNS64
        disk_space = DISK_SPACE
        environments = [
            PipEnvironment("yandex_tracker_client", version = "1.3", custom_parameters = ["--upgrade-strategy only-if-needed"]),
            PipEnvironment("startrek_client", version = "2.3.0", custom_parameters = ["--upgrade-strategy only-if-needed"])
        ]

    class Parameters(sdk2.Task.Parameters):
        ubuntu_version = create_ubuntu_selector()
        nodejs_version = create_node_selector()

        arcadia_url = sdk2.parameters.ArcadiaUrl(
            "Arcadia repository url",
            default_value = "arcadia-arc:/#trunk",
            required = True
        )
        app_dir_with_arc = sdk2.parameters.String(
            "App dir",
            default = "market/lilucrm/operator-window/webapp",
            required = True
        )
        base_url = sdk2.parameters.String(
            "Тестируемый хост",
            description = "Адрес машины, на который будут ходить тесты",
            required = True
        )
        grid_url = sdk2.parameters.String(
            "Селениум грид",
            default_value = "http://market@sw.yandex-team.ru:80/v0",
            required = True
        )
        issue = ReleaseTicket(
            "Релизный тикет",
            description = 'Пример: "OCRM-12345"'
        )

    def on_enqueue(self):
        super(LilucrmOperatorWindowAutotestsHermione, self).on_enqueue()
        setup_container(self)

    def _prepare_dirs(self):
        self.report_dir = self.path("report")
        self.stat_reporter_path = self.path("report/stat-reporter.json")

    def _prepare_env(self):
        os.environ["GRID_URL"] = self.Parameters.grid_url
        os.environ["BASE_URL"] = self.Parameters.base_url
        os.environ["REPORT_DIR"] = str(self.report_dir)
        os.environ["STAT_REPORTER_PATH"] = str(self.stat_reporter_path)
        os.environ["MAIN_USER_PASSWORD"] = sdk2.Vault.data(MAIN_PASSWORD_TOKEN_VAULT_KEY)
        os.environ["ADDITIONAL_USER_PASSWORD"] = sdk2.Vault.data(ADDITIONAL_PASSWORD_TOKEN_VAULT_KEY)

    def _test(self, app_path):
        rich_check_call(["bash", "-c", "npm install && npm run hermione:run:grid"], self, "test", cwd = app_path)

    def _create_report(self):
        resource = MARKET_AUTOTEST_REPORT(self, "Report", self.report_dir)
        sdk2.ResourceData(resource).ready()

        return resource

    def _send_startrek_report(self, resource):
        if not self.Parameters.issue:
            logging.debug("Release ticket not specified")
            return

        report_url = "{}/index.html".format(get_resource_link(resource)) if resource else ""
        oauth_token = sdk2.Vault.data(ST_OAUTH_TOKEN_VAULT_KEY)

        from startrek_client import Startrek
        st = Startrek(useragent = "robot-crm-tifa", token = oauth_token)
        issue = st.issues[self.Parameters.issue]
        task_url = get_task_link(self.id)

        with open(str(self.stat_reporter_path), 'r') as file:
            report_data = json.loads(file.read())

            if not report_data:
                logging.info('Empty report data in {}'.format(self.stat_reporter_path))
                return

            html_report = _build_test_statistic(
                tool = 'hermione',
                data = report_data,
                report_url = report_url or '{} [Отсутствует] '.format(task_url),
                task_url = task_url
            )

            issue.comments.create(text = html_report)

            logging.debug('Statistic has been sent to {}'.format(self.Parameters.issue))


    def on_prepare(self):
        self._prepare_dirs()
        self._prepare_env()

    def on_execute(self):
        super(LilucrmOperatorWindowAutotestsHermione, self).on_execute()
        with SetEnv(self, nodejs_version=self.Parameters.nodejs_version, arc_token_vault="robot-market-devexp-arc-token"):
            self._prepare_repos()

    def _prepare_repos(self):
        url = self.Parameters.arcadia_url
        use_arc = True
        with arcadia_sdk.mount_arc_path(url, use_arc_instead_of_aapi=use_arc) as arcadia_root:
            arcadia_path = os.path.join(arcadia_root, self.Parameters.app_dir_with_arc)

            try:
                self._test(arcadia_path)
            finally:
                report_resource = self._create_report()
                self._send_startrek_report(report_resource)
