import logging
import os
import tarfile

import sandbox.common.types.client as ctc

from sandbox import sdk2
from sandbox.sdk2.helpers import subprocess as sp

class LilucrmRunMigrations(sdk2.Task):

    """Runs java application for executing database migrations.
    Application package should contains specialized shell script to run application in migration-only mode.
    See example https://a.yandex-team.ru/arc/trunk/arcadia/market/lilucrm/operator-window/ya.make?rev=5535076#L93

    Task copies secrets to application properties file from the vault specified in parameters.
    """

    class Requirements(sdk2.Task.Requirements):
        client_tags = ctc.Tag.Group.LINUX

    class Parameters(sdk2.Task.Parameters):
        app_resource = sdk2.parameters.Resource('app resource')
        datasources = sdk2.parameters.Resource('environment datasources')
        environment = sdk2.parameters.String('environment')
        vault_owner = sdk2.parameters.String('vault passwords token owner', default='MARKET')
        vault_name = sdk2.parameters.String('vault with secret properties')
        workdir = sdk2.parameters.String('workdir')
        migration_script_path = sdk2.parameters.String('migration script path')


    def on_execute(self):
        self.workdir = self.path(self.Parameters.workdir).as_posix()

        self.prepare_resource(self.Parameters.app_resource)
        self.prepare_resource(self.Parameters.datasources)
        self.prepare_secrets()

        args = ' '.join([
            os.path.join(self.workdir, self.Parameters.migration_script_path),
            '--environment={}'.format(self.Parameters.environment),
            '--logdir={}'.format(self.log_path('java_logs'))
        ])

        with sdk2.helpers.ProcessLog(self, logger=logging.getLogger('migration')) as pl:
            return_code = sp.Popen(args, shell=True, stdout=pl.stdout, stderr=sp.STDOUT, cwd=self.workdir).wait()
            if return_code != 0:
                raise Exception('Return code is not zero: ' + str(return_code))

    def prepare_resource(self, resource):
        resource = sdk2.ResourceData(resource)
        resource_path = resource.path.as_posix()

        logging.info('resource path: %s', resource_path)

        self.extract_with_strip(resource_path)

    def prepare_secrets(self):
        secret_path = os.path.join(self.workdir, 'properties.d', '75_secret.properties')
        value = sdk2.Vault.data(self.Parameters.vault_owner, self.Parameters.vault_name)

        with open(secret_path, 'w') as f:
            f.write(value)

    def extract_with_strip(self, tar_path, strip=1):
        tar = tarfile.open(tar_path)
        for member in tar:
            split = member.name.split(os.path.sep)[strip:]
            if len(split) == 0:
                continue
            member.name = os.path.join(*split)
            tar.extract(member, path=self.workdir)
