from sandbox import sdk2
from sandbox.sdk2 import yav
import sandbox.common.types.client as ctc

import tarfile
import zipfile
import os
import os.path
import logging

class LilucrmUploadFrontStatic(sdk2.Task):

    """Upload frontend package with static files to MDS.
    Package must contain an archive with a folder containing files to upload.
    Archive name in package and folder name are specified in parameters "archive_filename" and "path_in_archive".
    The main html that links to other files should be uploaded last. Specify them in parameter "files_to_upload_at_end".
    """

    class Requirements(sdk2.Task.Requirements):
        client_tags = ctc.Tag.Group.LINUX

    class Parameters(sdk2.Task.Parameters):
        webapp_package_resource = sdk2.parameters.Resource('operator window webapp resource')
        archive_filename = sdk2.parameters.String('archive file name')
        path_in_archive = sdk2.parameters.String('path in archive')
        yav_s3_secret_id = sdk2.parameters.String('yav secret ID with s3 access key')
        mds_endpoint = sdk2.parameters.String('mds endpoint')
        mds_bucket = sdk2.parameters.String('mds bucket')
        mds_path_in_bucket = sdk2.parameters.String('mds path in bucket')
        files_to_upload_at_end = sdk2.parameters.List('files to upload at the end')

    def on_execute(self):
        self.extract_package()
        self.extract_archive()
        self.upload_to_mds()
        self.send_notification()

    #-------------------------------------------------------------------------------------------------------------------

    def extract_package(self):
        resource = sdk2.ResourceData(self.Parameters.webapp_package_resource)
        resource_path = resource.path.as_posix()
        logging.info('extract package: %s', resource_path)
        with tarfile.open(resource_path) as tar:
            archive = tar.getmember(self.Parameters.archive_filename)
            tar.extract(archive, '.')

    def extract_archive(self):
        logging.info('extract archive: %s', self.Parameters.archive_filename)
        with zipfile.ZipFile(self.Parameters.archive_filename) as archive:
            for filename in archive.namelist():
                if filename.startswith(self.Parameters.path_in_archive):
                    archive.extract(filename, '.')

    def upload_to_mds(self):
        s3_client = self.connect_to_s3()
        dir_path = self.Parameters.path_in_archive
        logging.info('upload files from: %s', dir_path)
        for root, dir_names, file_names in os.walk(dir_path):
            for file_name in file_names:
                if file_name not in self.Parameters.files_to_upload_at_end:
                    self.upload_file(os.path.join(root, file_name), s3_client)
        for file_name in self.Parameters.files_to_upload_at_end:
            self.upload_file(os.path.join(dir_path, file_name), s3_client)

    def upload_file(self, full_path, s3_client):
        rel_path = os.path.relpath(full_path, self.Parameters.path_in_archive)
        mds_path = self.Parameters.mds_path_in_bucket + '/' + rel_path
        logging.info('upload: %s', mds_path)
        s3_client.upload_file(full_path, self.Parameters.mds_bucket, mds_path)

    def connect_to_s3(self):
        import boto3 # see https://wiki.yandex-team.ru/sandbox/faq/#import-error
        s3_secret = self.get_s3_secret()
        session = boto3.session.Session(aws_access_key_id=s3_secret['AccessKeyId'],
                                        aws_secret_access_key=s3_secret['AccessSecretKey'])
        return session.client(service_name='s3', endpoint_url=self.Parameters.mds_endpoint, verify=False)

    def get_s3_secret(self):
        secret = yav.Secret(self.Parameters.yav_s3_secret_id)
        return secret.data()

    def send_notification(self):
        logging.info('send notification to backend')
