import logging
import json
import requests
import time
from sandbox import sdk2


class DataTransferUpload(sdk2.Task):
    TRANSFER_UPLOAD_URL = "https://cdc.n.yandex-team.ru/v1/transfer/upload"
    OPERATIONS_GET_URL = "https://cdc.n.yandex-team.ru/v1/operation/{id}"
    SECONDS_TO_SLEEP = 120

    class Parameters(sdk2.Task.Parameters):
        dataTransferId = sdk2.parameters.String("DataTransfer ID", required=True)
        tableName = sdk2.parameters.String("Table name", required=True)
        filterColumn = sdk2.parameters.String("Filter column (id, updated, ...)", default="id")
        minValue = sdk2.parameters.Integer("Min value", required=True, default=None)
        maxValue = sdk2.parameters.Integer("Max value", required=True, default=None)
        step = sdk2.parameters.Integer("Step", required=True, default=None)
        with sdk2.parameters.String("Threads number", required=True) as threadsNumber:
            threadsNumber.values["1"] = threadsNumber.Value(1, default=True)
            threadsNumber.values["2"] = threadsNumber.Value(2)
            threadsNumber.values["3"] = threadsNumber.Value(3)
            threadsNumber.values["4"] = threadsNumber.Value(4)

    def on_execute(self):
        self.__log_parameters()
        parameters = self.Parameters

        right = parameters.minValue
        needStop = False
        iteration = 0
        while not needStop:
            tables_data = []
            for threadIndex in range(int(parameters.threadsNumber)):
                left = right
                right += parameters.step
                if right >= parameters.maxValue:
                    right = parameters.maxValue + 1
                    needStop = True

                tables_data.append(self.__create_tables_data(left, right))

                if needStop:
                    break

            logging.info("Start uploading tables data: %s", json.dumps(tables_data))
            response = self.__upload(tables_data)
            if response.status_code == 200:
                operation_id = response.json().get("id")
                logging.info("Received operation ID: %s", operation_id)
                time.sleep(self.SECONDS_TO_SLEEP)
                while not self.__is_operation_done(operation_id, left):
                    logging.info("Operation is not finished. Sleep for %s seconds", str(self.SECONDS_TO_SLEEP))
                    time.sleep(self.SECONDS_TO_SLEEP)
                logging.info("Operation is done!")
            else:
                logging.info("Incorrect status %s", str(response.status_code))
                logging.error("%s must be reuploaded", left)

            iteration += 1
            time.sleep(self.SECONDS_TO_SLEEP)
        logging.info("Copying is finished, iteration count: %s", str(iteration))

    def __headers(self):
        return {
            "accept": "application/json",
            "Authorization": self.__getAuthToken(),
            "Content-Type": "application/json"
        }

    @staticmethod
    def __getAuthToken():
        secret = sdk2.yav.Secret("sec-01evs9qrnnfepsqax2j343256j")
        return secret.data()["data-transfer-token"]

    def __log_parameters(self):
        parameters = self.Parameters
        logging.info("Starting copying data through DataTransfer Upload")
        logging.info("DataTransfer ID: %s", parameters.dataTransferId)
        logging.info("Table name: %s", parameters.tableName)
        logging.info("Threads number: %s", str(parameters.threadsNumber))
        logging.info("Filter column: %s", parameters.filterColumn)
        logging.info("Min value: %s", str(parameters.minValue))
        logging.info("Max value: %s", str(parameters.maxValue))

    def __create_tables_data(self, left, right):
        return {
           "schema": "public",
           "name": self.Parameters.tableName,
           "filter": "{column} >= {left} and {column} < {right}".format(
               column=self.Parameters.filterColumn,
               left=str(left),
               right=str(right)
           )
       }

    def __is_operation_done(self, operation_id, left):
        response = requests.get(
            url=self.OPERATIONS_GET_URL.format(id=operation_id),
            headers=self.__headers(),
        )
        try:
            return response.json().get("done") is True
        except ValueError:
            logging.error("Cannot parse JSON from response %s", response)
            logging.error("%s must be reuploaded", left)
            return True

    def __upload(self, tables_data):
        return requests.post(
            url=self.TRANSFER_UPLOAD_URL,
            headers=self.__headers(),
            data=json.dumps({
                "transfer_id": self.Parameters.dataTransferId,
                "tables": tables_data
            })
        )
