import codecs
import logging

from sandbox.projects.market.logistics.MarketLogisticGatewayLoadGenerateAmmo.generate import make_ammo
from sandbox import sdk2
from sandbox.projects.market.logistics import MarketLogisticsLoadAmmoAsync
from sandbox.sandboxsdk.environments import PipEnvironment
from flow_to_url_parser import get_flow_to_path_for_delivery, get_flow_to_path_for_fulfillment

try:
    from urllib.parse import urlparse
    from urllib.parse import parse_qs
except ImportError:
    from urlparse import urlparse
    from urlparse import parse_qs
from datetime import datetime, timedelta


HEADERS = 'Accept:"application/json, application/*+json",Connection:"close",Host:"lgw.vs.market.yandex.net",' \
          'Accept-Encoding:"gzip",Content-Type:"application/json"'


class MarketLogisticGatewayGenerateAmmoForAsyncRequests(sdk2.Task):
    class Parameters(sdk2.Task.Parameters):
        required_requests = sdk2.parameters.Integer('Requests to load from YT count', default=100000, required=True)
        same_requests_in_result = sdk2.parameters.Integer('Max number of same type requests in result', default=1,
                                                          required = True)
        fulfillment_token = sdk2.parameters.String('Token of test fulfillment service',
                                                   default='e016f94486921fb4f843c0df9aca298075e53c6585c29156aae84dba51617fca',
                                                   required=True)
        delivery_token = sdk2.parameters.String('Token of test delivery service',
                                                default='xxxxxxxxxxxxxxxxxxxxxxFakeSdTokenxxxxxxxxxxxxxxxxxxxxxxxxxx',
                                                required=True)
        fulfillment_id = sdk2.parameters.Integer('Id of test fulfillment service', default=999104, required=True)
        delivery_id = sdk2.parameters.Integer('Id of test delivery service', default=136, required=True)
        mock_base_url = sdk2.parameters.String('Base url for mock server',
                                               default='http://partner-api-mock.tst.vs.market.yandex.net',
                                               required=True)
        yt_token = sdk2.parameters.String('YT token', required=True)
        task_flow_to_url_for_delivery = sdk2.parameters.String('String in format (taskFlow, taskControllerUrl)[;(taskFlow, taskControllerUrl)]',
                                                               default='(DS_CREATE_ORDER, order/create);'
                                                               # TODO: return updateOrder for ammo generating:
                                                               #  https://st.yandex-team.ru/DELIVERY-14826.
                                                                       # '(DS_UPDATE_ORDER, order/update);'
                                                                       '(DS_GET_LABELS, labels/get);'
                                                                       '(DS_UPDATE_RECIPIENT, recipient/update);'
                                                                       '(DS_UPDATE_ORDER_DELIVERY_DATE, order-delivery-date/update);'
                                                                       '(DS_CREATE_INTAKE, intake/create);'
                                                                       '(DS_CANCEL_ORDER, order/cancel);'
                                                                       '(DS_CANCEL_PARCEL, parcel/cancel);'
                                                                       '(DS_CANCEL_ORDER_TRACK, order/track/cancel);'
                                                                       '(DS_CREATE_REGISTER, register/create)',
                                                               required=True)
        task_flow_to_url_for_fulfillment = sdk2.parameters.String('String in format (taskFlow, taskControllerUrl)[;(taskFlow, taskControllerUrl)]',
                                                                  default='(FF_CREATE_INBOUND, inbound/create);'
                                                                          '(FF_CANCEL_INBOUND, inbound/cancel);'
                                                                          '(FF_CREATE_ORDER, order/create);'
                                                                          '(FF_CANCEL_ORDER, order/cancel);'
                                                                  # TODO: return updateOrder for ammo generating:
                                                                  #  https://st.yandex-team.ru/DELIVERY-14826.
                                                                          # '(FF_UPDATE_ORDER, order/update);'
                                                                          '(FF_CREATE_OUTBOUND, outbound/create);'
                                                                          '(FF_CANCEL_OUTBOUND, outbound/cancel);'
                                                                          '(FF_CREATE_TRANSFER, transfer/create)',
                                                                  required=True)
        message_length_thresholds = sdk2.parameters.Dict('Thresholds for max lengths of messages for given task flows',
                                                         default={
                                                             'DS_CREATE_INTAKE': 1150,
                                                             'DS_CREATE_ORDER': 4500,
                                                             'DS_CREATE_REGISTER': 20000,
                                                             'DS_CREATE_SELFEXPORT': 1100,
                                                             'DS_GET_ATTACHED_DOCS': 850,
                                                             'DS_GET_LABELS': 200,
                                                             'DS_PUT_REFERENCE_WAREHOUSES': 1000,
                                                             'DS_UPDATE_ORDER': 5000,
                                                             'DS_UPDATE_ORDER_DELIVERY_DATE': 250,
                                                             'FF_CANCEL_INBOUND': 150,
                                                             'FF_CANCEL_OUTBOUND': 150,
                                                             'FF_CREATE_INBOUND': 30000,
                                                             'FF_CREATE_ORDER': 5000,
                                                             'FF_CREATE_OUTBOUND': 17500,
                                                             'FF_CREATE_TRANSFER': 400,
                                                             'FF_PUT_REFERENCE_ITEMS': 105000
                                                         }, required=True)
        message_length_default_threshold = sdk2.parameters.Integer('Default threshold for max length of message',
                                                                   default=5000, required=True)

    class Requirements(sdk2.Requirements):
        disk_space = 1024 * 5
        environments = (
            PipEnvironment('yandex-yt'),
            PipEnvironment('yandex-yt-yson-bindings'),
            PipEnvironment('yandex-yt-yson-bindings-skynet'),
        )

    def on_execute(self):
        ammo = self.generate_ammo()

        data = sdk2.ResourceData(MarketLogisticsLoadAmmoAsync(self, 'Logistic gateway load ammo', 'lgw_shooting_ammo_async.txt'))
        with codecs.open('lgw_shooting_ammo_async.txt', 'w', encoding='utf-8') as f:
            f.write(ammo)
        data.ready()

    def get_data_from_yt(self):
        import yt.wrapper
        logging.info('Start fetching requests data from YT')
        yt.wrapper.config.set_proxy("hahn")
        yt.wrapper.config['token'] = self.Parameters.yt_token
        yesterday = datetime.now() - timedelta(days=1)
        table = yt.wrapper.TablePath("//logs/market-logistic-gateway-sqs-requests-log/1d/"
                                     + yesterday.strftime('%Y-%m-%d'), start_index=0,
                                     end_index=self.Parameters.required_requests)
        return yt.wrapper.read_table(table)

    def get_requests_to_generate_ammo_from_yt(self):
        all_data = self.get_data_from_yt()
        requests_to_generate_ammo = []
        method_to_count = {}
        flow_to_path_for_delivery = get_flow_to_path_for_delivery(self.Parameters.task_flow_to_url_for_delivery)
        flow_to_path_for_fulfillment = get_flow_to_path_for_fulfillment(self.Parameters.task_flow_to_url_for_fulfillment)
        for request in all_data:
            if self.is_empty(request['messageBody']):
                continue
            flow = request['taskFlow']
            if flow not in flow_to_path_for_delivery and flow not in flow_to_path_for_fulfillment:
                logging.info('Unknown flow: ' + flow + ', maybe you should add it to parameters')
                continue
            message_length_threshold = int(self.Parameters.message_length_thresholds.get(
                flow, self.Parameters.message_length_default_threshold))
            message_length = len(request['messageBody'])
            if message_length > message_length_threshold:
                continue
            count = 0
            if flow in method_to_count:
                count = method_to_count[flow]
            if count == self.Parameters.same_requests_in_result:
                continue
            count += 1
            method_to_count[flow] = count
            if flow in flow_to_path_for_delivery:
                path = flow_to_path_for_delivery[flow]
            else:
                path = flow_to_path_for_fulfillment[flow]
            logging.info('The length of messageBody of added ammo equals to ' + str(message_length) +
                         ' which is not larger than threshold: ' + str(message_length_threshold))
            requests_to_generate_ammo.append((request['messageBody'], path))
        return requests_to_generate_ammo

    def is_empty(self, body):
        return body is None or len(body) == 0

    def generate_ammo(self):
        ammo = ''
        yt_data = self.get_requests_to_generate_ammo_from_yt()
        for element in yt_data:
            body = element[0]
            path = element[1]
            ammo_for_request = make_ammo('POST', path, path, HEADERS, body,
                                         self.Parameters.fulfillment_token, self.Parameters.delivery_token,
                                         self.Parameters.fulfillment_id, self.Parameters.delivery_id)
            ammo += ammo_for_request
        return ammo

