import logging
import xml.etree.ElementTree as ET
from datetime import datetime, timedelta

import requests

from sandbox import sdk2
from sandbox.sandboxsdk.environments import PipEnvironment


class MarketLogisticGatewayGenerateMocksForAsyncRequests(sdk2.Task):
    class Parameters(sdk2.Task.Parameters):
        mock_base_url = sdk2.parameters.String('Base url for mock server',
                                               default='http://partner-api-mock.tst.vs.market.yandex.net/',
                                               required=True)
        yt_token = sdk2.parameters.String('YT token', required=True)
        methods_to_skip_string = sdk2.parameters.String('Methods, for which mocks shouldn\'t be generated '
                                                        '(for example, sync methods or methods with problems)',
                                                        default='getReferencePickupPoints,getOrdersDeliveryDate,'
                                                                'getOrderHistory,getOrdersStatus,getOrder,'
                                                                'getTransferDetails,getTransferHistory,'
                                                                'getTransfersStatus,getInboundDetails,'
                                                                'getInboundHistory,getInboundsStatus,getStocks,'
                                                                'getReferenceItems,getExpirationItems,'
                                                                'getOutboundHistory,getOutboundDetails,'
                                                                'getOutboundsStatus,'
                                                        # TODO: remove updateOrder from this list:
                                                        #  https://st.yandex-team.ru/DELIVERY-14826
                                                                'updateOrder',
                                                        required=True)
        delivery_methods_string = sdk2.parameters.String('Delivery methods, there aren\'t such methods for fulfillment',
                                                         default='getLabels,updateOrderDeliveryDate,updateRecipient,'
                                                                 'createIntake,cancelOrderTrack,cancelParcel,'
                                                                 'createRegister,createSelfExport',
                                                         required=True)
        fulfillment_methods_string = sdk2.parameters.String('Fulfillment methods, there aren\'t such methods for delivery',
                                                            default='createInbound,createTransfer,getInboundDetails,'
                                                                    'cancelInbound,createOutbound,cancelOutbound',
                                                            required=True)
        delivery_response_body_length_thresholds = sdk2.parameters.Dict('Thresholds for max lengths of response bodies '
                                                                        'for given DS request types',
                                                                        default={
                                                                            'createIntake': 350,
                                                                            'createOrder': 350,
                                                                            'createRegister': 400,
                                                                            'createSelfExport': 350,
                                                                            'getAttachedDocs': 12000,
                                                                            'getLabels': 400,
                                                                            'putReferenceWarehouses': 350,
                                                                            'updateOrder': 700,
                                                                            'updateOrderDeliveryDate': 350
                                                                        }, required=True)
        fulfillment_response_body_length_thresholds = sdk2.parameters.Dict('Thresholds for max lengths of response '
                                                                           'bodies for given FF request types',
                                                                           default={
                                                                               'createInbound': 350,
                                                                               'createIntake': 400,
                                                                               'createOrder': 400,
                                                                               'createOutbound': 400,
                                                                               'createRegister': 22500,
                                                                               'createSelfExport': 300,
                                                                               'createTransfer': 350,
                                                                               'putReferenceItems': 250
                                                                           }, required=True)
        delivery_response_body_length_default_threshold = sdk2.parameters.Integer('Default threshold for max length of'
                                                                                  'response body for DS',
                                                                                  default=9000, required=True)
        fulfillment_response_body_length_default_threshold = sdk2.parameters.Integer('Default threshold for max length'
                                                                                     'of response body for FF',
                                                                                     default=40000, required=True)
        page_size = sdk2.parameters.Integer('Page size to read from YT', default=10000, required=True)
        required_requests = sdk2.parameters.Integer('Requests to load from YT count', default=1500000, required=True)

    class Requirements(sdk2.Requirements):
        disk_space = 1024 * 5
        environments = (
            PipEnvironment('yandex-yt'),
            PipEnvironment('yandex-yt-yson-bindings'),
            PipEnvironment('yandex-yt-yson-bindings-skynet'),
        )

    def on_execute(self):
        self.get_requests_to_generate_ammo_from_yt()

    def get_data_from_yt(self, from_index, to_index):
        import yt.wrapper
        logging.info('Start fetching requests data from YT')
        yt.wrapper.config.set_proxy("hahn")
        yt.wrapper.config['token'] = self.Parameters.yt_token
        yesterday = datetime.now() - timedelta(days=1)
        table = yt.wrapper.TablePath("//logs/market-logistic-gateway-partner-request-log/1d/"
                                     + yesterday.strftime('%Y-%m-%d'), start_index=from_index,
                                     end_index=to_index)
        return yt.wrapper.read_table(table)

    def get_requests_to_generate_ammo_from_yt(self):
        methods_to_skip = self.Parameters.methods_to_skip_string.split(',')
        delivery_methods = self.Parameters.delivery_methods_string.split(',')
        fulfillment_methods = self.Parameters.fulfillment_methods_string.split(',')
        from_index = 0
        to_index = self.Parameters.page_size
        types_was_before = []
        while True:
            all_data = self.get_data_from_yt(from_index, to_index)
            need_break = True
            for dict in all_data:
                need_break = False
                if dict['type'] == 'request':
                    continue
                if dict['status_code'] != '200':
                    continue
                body = dict['body']
                try:
                    root = ET.fromstring(body)
                except ET.ParseError:
                    continue
                response = root.find('response')
                if response is None:
                    continue
                if 'type' not in response.attrib:
                    continue
                request_state = root.find('requestState')
                if request_state is None:
                    continue
                is_error = request_state.findtext('isError')
                if not is_error:
                    errors = request_state.find('errors')
                    if errors and list(errors.iter()):
                        continue
                    error_codes = request_state.find('errorCodes')
                    if error_codes and list(error_codes.iter()):
                        continue
                elif is_error != 'false':
                    continue
                type = response.attrib['type']
                if type in methods_to_skip:
                    continue
                if type in types_was_before:
                    continue
                is_delivery = type in delivery_methods or self.is_delivery_response(response)
                body_length_threshold = int(self.Parameters.delivery_response_body_length_thresholds.get(
                    type, self.Parameters.delivery_response_body_length_default_threshold) \
                    if is_delivery \
                    else self.Parameters.fulfillment_response_body_length_thresholds.get(
                    type, self.Parameters.fulfillment_response_body_length_default_threshold))
                body_length = len(body)
                if body_length > body_length_threshold:
                    continue
                types_was_before.append(type)
                request = ET.Element('request')
                request.attrib['type'] = type
                request_string = ET.tostring(request, encoding='utf-8').decode('utf-8')
                uniq = root.find('uniq')
                if uniq is not None:
                    uniq.text = '${uniq}'
                else:
                    uniq = ET.SubElement(root, 'uniq')
                    uniq.text = '${uniq}'
                hash = root.find('hash')
                if hash is not None:
                    hash.text = '${hash}'
                else:
                    hash = ET.SubElement(root, 'hash')
                    hash.text = '${hash}'
                response_string = ET.tostring(root, encoding='utf-8').decode('utf-8')
                item_for_post = self.create_item_for_post(request_string, response_string, 'xml')
                content_type_header = {'Content-Type': 'application/xml'}
                type = ('delivery-' if is_delivery else 'fulfillment-') + type
                mock_url = self.Parameters.mock_base_url + 'stress-' + type + '/mock'
                logging.info('The length of body of added mock equals to ' + str(body_length) +
                             ' which is not larger than threshold: ' + str(body_length_threshold))
                requests.post(mock_url, data=item_for_post, headers=content_type_header)
                logging.info(request_string)
                logging.info(response_string)
                logging.info(mock_url)
            from_index += self.Parameters.page_size
            to_index += self.Parameters.page_size
            logging.info(from_index)
            if need_break or from_index >= self.Parameters.required_requests:
                break

    def is_delivery_response(self, response):
        for elem in response:
            if elem.tag == 'deliveryId':
                return True
            if self.is_delivery_response(elem):
                return True
        return False

    def create_item_for_post(self, request_body, response_body, content_type):
        root = ET.Element('root')
        disposable = ET.SubElement(root, "disposable")
        disposable.text = 'false'
        in_element = ET.SubElement(root, "in")
        in_element.text = request_body
        out = ET.SubElement(root, "out")
        out.text = response_body
        content_type_element = ET.SubElement(root, "contentType")
        content_type_element.text = content_type
        contains_content_patterns = ET.SubElement(root, "containsContentPatterns")
        contains_content_patterns.text = 'false'
        return ET.tostring(root, encoding='utf-8')
