import codecs
import logging

from sandbox import sdk2
from sandbox.projects.market.logistics import MarketLogisticsLoadAmmoSync, MarketLogisticsGenerateEstimatedMocksSync
from sandbox.sandboxsdk.environments import PipEnvironment
from sandbox.projects.market.logistics.MarketLogisticGatewayLoadGenerateAmmo.generate import make_ammo
from sandbox.projects.market.logistics.MarketLogisticGatewayLoadGenerateAmmo.prepare_mock import prepare_mock
from tag_to_mock_url_parser import get_tag_to_mock_url_mapping

try:
    from urllib.parse import urlparse
    from urllib.parse import parse_qs
except ImportError:
    from urlparse import urlparse
    from urlparse import parse_qs
from datetime import datetime, timedelta
from xml.dom import minidom


class MarketLogisticGatewayGenerateAmmoForSyncRequests(sdk2.Task):
    class Parameters(sdk2.Task.Parameters):
        required_requests = sdk2.parameters.Integer('Requests to load from YT count', default=100000, required=True)
        same_requests_in_result = sdk2.parameters.Integer('Max number of same type requests in result', default=1,
                                                          required = True)
        fulfillment_token = sdk2.parameters.String('Token of test fulfillment service',
                                                   default='e016f94486921fb4f843c0df9aca298075e53c6585c29156aae84dba51617fca',
                                                   required=True)
        delivery_token = sdk2.parameters.String('Token of test delivery service',
                                                default='xxxxxxxxxxxxxxxxxxxxxxFakeSdTokenxxxxxxxxxxxxxxxxxxxxxxxxxx',
                                                required=True)
        fulfillment_id = sdk2.parameters.Integer('Id of test fulfillment service', default=999104, required=True)
        delivery_id = sdk2.parameters.Integer('Id of test delivery service', default=136, required=True)
        mock_base_url = sdk2.parameters.String('Base url for mock server',
                                               default='http://partner-api-mock.tst.vs.market.yandex.net',
                                               required=True)
        yt_token = sdk2.parameters.String('YT token', required=True)
        query_gateway_method_to_mock_path = sdk2.parameters.String('String in format (methodTag, mockUrl)[;(methodTag, mockUrl)]',
                                                                   default='(/fulfillment/query-gateway/pushStocks, stocks);'
                                                                           '(/fulfillment/quety-gateway/pushOrdersStatusesChanged, orders/status/push);'
                                                                           '(/delivery/query-gateway/pushOrdersStatusesChanged, orders/status/push)',
                                                                   required=True)
        request_length_thresholds = sdk2.parameters.Dict('Thresholds for max lengths of request bodies for given URIs',
                                                         default={
                                                             '/delivery/getOrderHistory': 150,
                                                             '/delivery/getOrdersDeliveryDate': 100,
                                                             '/delivery/getOrdersStatus': 7000,
                                                             '/delivery/getReferencePickupPoints': 150,
                                                             '/delivery/getReferenceWarehouses': 100,
                                                             '/delivery/getTransactionsOrders': 200,
                                                             '/delivery/query-gateway': 400,
                                                             '/fulfillment/getExpirationItems': 1050,
                                                             '/fulfillment/getInboundDetails': 150,
                                                             '/fulfillment/getInboundDetailsXDoc': 150,
                                                             '/fulfillment/getInboundHistory': 150,
                                                             '/fulfillment/getInboundsStatus': 450,
                                                             '/fulfillment/getOrderHistory': 150,
                                                             '/fulfillment/getOrdersStatus': 7500,
                                                             '/fulfillment/getOutboundDetails': 150,
                                                             '/fulfillment/getOutboundHistory': 150,
                                                             '/fulfillment/getOutboundsStatus': 450,
                                                             '/fulfillment/getReferenceItems': 900,
                                                             '/fulfillment/getStocks': 100,
                                                             '/fulfillment/getTransferDetails': 150,
                                                             '/fulfillment/getTransferHistory': 150,
                                                             '/fulfillment/getTransfersStatus': 200,
                                                             '/fulfillment/query-gateway': 70000
                                                         }, required=True)
        request_length_default_threshold = sdk2.parameters.Integer('Default threshold for max length of request body',
                                                                   default=1150, required=True)
        response_length_thresholds = sdk2.parameters.Dict('Thresholds for max lengths of response bodies '
                                                          'for given URIs',
                                                          default={
                                                              '/delivery/getOrderHistory': 4500,
                                                              '/delivery/getOrdersDeliveryDate': 250,
                                                              '/delivery/getOrdersStatus': 21500,
                                                              '/delivery/getReferencePickupPoints': 150,
                                                              '/delivery/getReferenceWarehouses': 16000000,
                                                              '/delivery/getTransactionsOrders': 25,
                                                              '/delivery/query-gateway': 200,
                                                              '/fulfillment/getExpirationItems': 9500,
                                                              '/fulfillment/getInboundDetails': 15000,
                                                              '/fulfillment/getInboundDetailsXDoc': 200,
                                                              '/fulfillment/getInboundHistory': 350,
                                                              '/fulfillment/getInboundsStatus': 800,
                                                              '/fulfillment/getOrderHistory': 450,
                                                              '/fulfillment/getOrdersStatus': 18500,
                                                              '/fulfillment/getOutboundDetails': 4500,
                                                              '/fulfillment/getOutboundHistory': 550,
                                                              '/fulfillment/getOutboundsStatus': 1000,
                                                              '/fulfillment/getReferenceItems': 21000,
                                                              '/fulfillment/getStocks': 80000,
                                                              '/fulfillment/getTransferDetails': 6000,
                                                              '/fulfillment/getTransferHistory': 5000,
                                                              '/fulfillment/getTransfersStatus': 350,
                                                              '/fulfillment/query-gateway': 250
                                                          }, required=True)
        response_length_default_threshold = sdk2.parameters.Integer('Default threshold for max length of response body',
                                                                    default=13500, required=True)

    class Requirements(sdk2.Requirements):
        disk_space = 1024 * 5
        environments = (
            PipEnvironment('yandex-yt'),
            PipEnvironment('yandex-yt-yson-bindings'),
            PipEnvironment('yandex-yt-yson-bindings-skynet'),
        )

    def on_execute(self):
        generated = self.generate_ammo()
        ammo = generated[0]
        partner_actions = generated[1]

        data = sdk2.ResourceData(MarketLogisticsLoadAmmoSync(self, 'Logistic gateway load ammo', 'lgw_shooting_ammo_sync.txt'))
        with codecs.open('lgw_shooting_ammo_sync.txt', 'w', encoding='utf-8') as f:
            f.write(ammo)
        data.ready()

        data2 = sdk2.ResourceData(MarketLogisticsGenerateEstimatedMocksSync(self,
                                                                            'Estimated partner requests and responses that should be inserted to mocks',
                                                                            'estimated_partner_requests_and_responses_sync.txt'))
        with codecs.open('estimated_partner_requests_and_responses_sync.txt', 'w', encoding='utf-8') as f:
            f.writelines(partner_actions)
        data2.ready()

    def get_data_from_yt(self):
        import yt.wrapper
        logging.info('Start fetching requests data from YT')
        yt.wrapper.config.set_proxy("hahn")
        yt.wrapper.config['token'] = self.Parameters.yt_token
        yesterday = datetime.now() - timedelta(days=1)
        table = yt.wrapper.TablePath("//logs/market-logistic-gateway-incoming-requests-log/1d/"
                                     + yesterday.strftime('%Y-%m-%d'), start_index=0,
                                     end_index=self.Parameters.required_requests)
        return yt.wrapper.read_table(table)

    def get_method_name_from_xml(self, xml):
        parsed_xml = minidom.parseString(xml)
        request_tag = parsed_xml.getElementsByTagName('request')[0]
        return request_tag.attributes['type'].value

    def is_delivery_or_fulfillment_path(self, path):
        return path.startswith('/delivery') or path.startswith('/fulfillment')

    def get_successful_requests_from_yt(self):
        all_data = self.get_data_from_yt()
        successful_requests = []
        method_to_count = {}
        for request in all_data:
            if request['status_code'] != '200':
                continue
            if self.is_empty(request['request_body']):
                continue
            if self.is_empty(request['response_body']):
                continue
            parsed_url = urlparse(request['uri'])
            path = parsed_url.path
            if not self.is_delivery_or_fulfillment_path(path):
                continue
            request_length_threshold = int(self.Parameters.request_length_thresholds.get(
                path, self.Parameters.request_length_default_threshold))
            request_length = len(request['request_body'])
            if request_length > request_length_threshold:
                continue
            response_length_threshold = int(self.Parameters.response_length_thresholds.get(
                path, self.Parameters.response_length_default_threshold))
            response_length = len(request['response_body'])
            if response_length > response_length_threshold:
                continue
            tag = path
            if path == '/fulfillment/query-gateway' or path == '/delivery/query-gateway':
                tag = tag + '/' + self.get_method_name_from_xml(request['request_body'])
            count = 0
            if tag in method_to_count:
                count = method_to_count[tag]
            if count == self.Parameters.same_requests_in_result:
                continue
            count += 1
            method_to_count[tag] = count
            logging.info('The length of request_body of added ammo equals to ' + str(request_length) +
                         ' which is not larger than threshold: ' + str(request_length_threshold))
            logging.info('The length of response_body of added ammo equals to ' + str(response_length) +
                         ' which is not larger than threshold: ' + str(response_length_threshold))
            successful_requests.append((request, parsed_url, tag))
        return successful_requests

    def is_empty(self, body):
        return body is None or len(body) == 0

    def generate_ammo(self):
        ammo = ''
        partner_requests_responses = []
        yt_data = self.get_successful_requests_from_yt()
        tag_to_mock_url_mapping = get_tag_to_mock_url_mapping(self.Parameters.query_gateway_method_to_mock_path)
        for element in yt_data:
            request = element[0]
            parsed_url = element[1]
            tag = element[2]
            mock_url = None
            if tag in tag_to_mock_url_mapping:
                mock_url = tag_to_mock_url_mapping[tag]
            partner_actions = prepare_mock(parsed_url.path, request['request_body'], request['response_body'],
                                           self.Parameters.mock_base_url, mock_url, self.Parameters.delivery_id,
                                           self.Parameters.fulfillment_id)
            ammo_for_request = make_ammo(request['method'], parsed_url.path, tag, request['request_headers'],
                                         request['request_body'], self.Parameters.fulfillment_token,
                                         self.Parameters.delivery_token, self.Parameters.fulfillment_id,
                                         self.Parameters.delivery_id)
            partner_requests_responses.append(partner_actions)
            ammo += ammo_for_request
        return ammo, partner_requests_responses

