import logging
from sandbox import sdk2
from sandbox.sdk2.helpers import subprocess
from sandbox.sdk2 import Task
from sandbox.common.types import client as ctc
from sandbox.common.types import resource as ctr
from sandbox.common.errors import SubprocessError
from sandbox.projects.common import binary_task
from sandbox.projects.common.arcadia import sdk
from sandbox.projects.common.environments import SandboxJavaJdkEnvironment

LIQUIBASE_RESOURCE_ID = 709114857
PG_JDBC_RESOURCE_ID = 3267404037


class MarketMultitestLiquibaseUpdate(binary_task.LastBinaryTaskRelease, Task):
    """
    Task runs liquibase update.
    """

    class Requirements(sdk2.Task.Requirements):
        environments = [SandboxJavaJdkEnvironment('1.8.0')]
        client_tags = ctc.Tag.Group.LINUX

    class Parameters(sdk2.Task.Parameters):
        ext_params = binary_task.binary_release_parameters(stable=True)

        url = sdk2.parameters.String('Connection string', required=True)
        username = sdk2.parameters.String('Username', required=True)
        password_vault_name = sdk2.parameters.String('Password vault record name', required=True)
        password_vault_owner = sdk2.parameters.String('Password vault record owner', required=True)
        branch = sdk2.parameters.String('Branch', required=False, default='trunk')
        changelog_file = sdk2.parameters.String('Changelog file path', required=True)
        additional_options = sdk2.parameters.List('Additional options (e.g.: --delimiter=|)', default=[])
        with sdk2.parameters.String("Liquibase command", multiline=True) as command:
            command.values.updateSQL = command.Value(default=True)
            command.values.update = None
            command.values.rollbackCount = None
        jdbc_resource_id = sdk2.parameters.String('Jdbc driver resource id', required=False,
                                                  default=PG_JDBC_RESOURCE_ID)
        liquibase_jar_resource_id = sdk2.parameters.String('Liquibase jar resource id', required=False,
                                                           default=LIQUIBASE_RESOURCE_ID)
        additional_arguments = sdk2.parameters.List('Additional arguments', default=[])
        additional_java_properties = sdk2.parameters.List('Additional java properties', default=[])

    @property
    def binary_executor_query(self):
        return {
            "attrs": {
                "task_type": "MARKET_MULTITEST_LIQUIBASE_UPDATE",
                "released": self.Parameters.binary_executor_release_type
            },
            "state": [ctr.State.READY]
        }

    @staticmethod
    def get_resource_path(resource_id):
        return str(sdk2.ResourceData(sdk2.Resource.find(id=resource_id).first()).path)

    def on_execute(self):
        url = self.Parameters.url
        username = self.Parameters.username
        password = sdk2.Vault.data(self.Parameters.password_vault_owner, self.Parameters.password_vault_name)
        branch = self.Parameters.branch
        changelog_file = self.Parameters.changelog_file
        additional_options = self.Parameters.additional_options
        command = self.Parameters.command
        additional_arguments = self.Parameters.additional_arguments
        additional_java_properties = self.Parameters.additional_java_properties
        jdbc_resource_id = self.Parameters.jdbc_resource_id
        liquibase_jar_resource_id = self.Parameters.liquibase_jar_resource_id
        if not jdbc_resource_id:
            jdbc_resource_id = PG_JDBC_RESOURCE_ID
        if not liquibase_jar_resource_id:
            liquibase_jar_resource_id = LIQUIBASE_RESOURCE_ID

        with sdk.mount_arc_path('arcadia-arc:/#{}'.format(branch)) as arcadia_root:
            logging.info("Arcadia path: {}".format(arcadia_root))
            liquibase_path = self.get_resource_path(liquibase_jar_resource_id)
            classpath = self.get_resource_path(jdbc_resource_id)
            with sdk2.helpers.ProcessLog(self, logger='liquibase') as pl:
                retcode = subprocess.Popen(
                    'java {additional_java_properties} -jar {liquibase} --classpath={classpath} --changeLogFile={changelog_file} '
                    '--url="{url}" --username={username} --password={password} {additional_options} {command} {additional_arguments}'.format(
                        additional_java_properties=' '.join(additional_java_properties),
                        liquibase=liquibase_path,
                        classpath=classpath,
                        changelog_file=arcadia_root + changelog_file,
                        url=url,
                        username=username,
                        password=password,
                        additional_options=' '.join(additional_options),
                        command=command,
                        additional_arguments=' '.join(additional_arguments)
                    ),
                    shell=True, stdout=pl.stdout, stderr=subprocess.STDOUT
                ).wait()
                if retcode:
                    raise SubprocessError('Java process exited with non-zero return code {}'.format(retcode))
