# coding=utf-8

import json
import logging

import sandbox.common.types.misc as ctm
import sandbox.common.types.client as ctc
from sandbox import sdk2
from sandbox.projects.sandbox_ci.utils.request import send_request
from sandbox.sdk2 import Task
from sandbox.projects.common import binary_task
from sandbox.common.types import resource as ctr
from sandbox.projects.market.wms.utils_py3 import mt_utils

ENVIRONMENT_LINK_MESSAGE = "Ссылка на среду: {} (запуск не найден)"
LAUNCH_LINK_MESSAGE = "Ссылка на запуск: {}"
NOTIFICATION = """**Запустился пайплайн по созданию мультитестинговой среды**
        {}
        Балансер MDB: {}
        Балансер LOM: {}
        """

LOGISTICS_MT_URL = "https://{project}--{name}-{app}.demofslb.market.yandex-team.ru"
TSUM_LAUNCH_URL = "{tsum_url}/launch/{launch_id}"


class StartLogisticsMT(binary_task.LastBinaryTaskRelease, Task):
    """
    Таска для запуска и перезапуска мультитестинга логистики.
    """

    class Requirements(Task.Requirements):
        dns = ctm.DnsType.DNS64
        client_tags = ctc.Tag.MULTISLOT

    class Parameters(Task.Parameters):
        ext_params = binary_task.binary_release_parameters(stable=True)

        with sdk2.parameters.Output():
            lom_mt_url = sdk2.parameters.String("MT-Lom balancer")
            mdb_mt_url = sdk2.parameters.String("MT-Mdb balancer")

        ticket_id = sdk2.parameters.String("Тикет", required=True)
        branch = sdk2.parameters.String("Ветка")
        arcadia_url = sdk2.parameters.ArcadiaUrl("Arcadia repository url", default="arcadia-arc:/#trunk", required=True)
        tsum_project = sdk2.parameters.String("Название проекта", default="logistics", required=True)
        tsum_pipeline_type = sdk2.parameters.String("Тип пайплайна", default="Multitest-logistics",
                                                    required=True)
        robot_secret = sdk2.parameters.YavSecret("Секрет робота", default="sec-01cqm962khgab2bexrcds931g0",
                                                 required=True)
        arcanum_review_id = sdk2.parameters.Integer("Arcanum PR id", default=None)

    @property
    def binary_executor_query(self):
        return {
            "attrs": {
                "task_type": "START_LOGISTICS_MT",
                "released": self.Parameters.binary_executor_release_type
            },
            "state": [ctr.State.READY]
        }

    @property
    def _tsum_mt_name(self):
        return mt_utils.tsum_mt_name(self.Parameters.ticket_id, self.Parameters.branch)

    @property
    def _issue_name(self):
        return self.Parameters.ticket_id

    @property
    def _tsum_api_url(self):
        return mt_utils.tsum_api_url(self.Parameters.tsum_project, self._tsum_mt_name)

    @property
    def _tsum_page_url(self):
        return mt_utils.tsum_page_url(self.Parameters.tsum_project, self._tsum_mt_name)

    @property
    def _tsum_exists_mt_page_url(self):
        return mt_utils.tsum_exists_mt_page_url(self.Parameters.tsum_project, self._tsum_mt_name)

    @property
    def _robot_secret(self):
        return self.Parameters.robot_secret

    @property
    def _tsum_headers(self):
        return {
            "Authorization": self._robot_secret.data()["tsum-token"],
            "Content-Type": "application/json"
        }

    def _tsum_launch_url(self, tsum_url, launch_id):
        return TSUM_LAUNCH_URL.format(tsum_url=tsum_url, launch_id=str(launch_id))

    def _app_balancer_url(self, app):
        return LOGISTICS_MT_URL.format(project=self.Parameters.tsum_project, name=self._tsum_mt_name, app=app)

    def _notify_pr(self, message):
        if not self.Parameters.arcanum_review_id:
            return

        import vcs.arcanum as arcanum
        import vcs.arcanum.comments as comments

        logging.info("Sending comment to arcanum")
        try:
            arcanum_client = arcanum.Arcanum(auth=self._robot_secret.data()["arcanum-token"])

            comments_api = comments.CommentsApi(arcanum_client)
            if self.Parameters.arcanum_review_id:
                comments_api.post_review_requests_comments(
                    review_request_id=self.Parameters.arcanum_review_id,
                    content=message,
                    draft=False,
                )
        except Exception as e:
            logging.error("Can't post comment into arcanum review")
            logging.error(e.message)

    def _send_startrek_report(self, message):
        logging.info("Sending comment to startrek")
        try:
            from startrek_client import Startrek
            st = Startrek(useragent="robot-infra-deli", token=self._robot_secret.data()["startrek-token"])
            issue = st.issues[self._issue_name]
            issue.comments.create(text=message)
        except Exception as e:
            logging.error("Can't send startrek report")
            logging.error(e.message)

        return

    def _start_mt(self):
        data = {
            "name": self._tsum_mt_name,
            "type": "USE_EXISTING_PIPELINE",
            "pipelineId": self.Parameters.tsum_pipeline_type,
            "resources": {
                "fd4f152f-9f11-4b91-8a35-e58503b6bdf6": {
                    "tickets": [self.Parameters.ticket_id]
                },
                "c549477a-867a-483f-947b-5fd64feef256": {
                    "ref": self.Parameters.arcadia_url
                }
            }
        }

        res = send_request("post", self._tsum_api_url, headers=self._tsum_headers, data=json.dumps(data))
        try:
            res.raise_for_status()
        except Exception as e:
            logging.error("Starting multitesting in TSUM failed. Request body: {}".format(data))
            logging.error("TSUM response: {}".format(res.json()))
            logging.error(e.message)
            raise e

        try:
            assert res.json()["name"] == self._tsum_mt_name
        except AssertionError as e:
            logging.error(
                "Incorrect multitesting name in TSUM response: \"{}\", expected: \"{}\"".format(res.json()["name"],
                                                                                                self._tsum_mt_name))
            raise e

        return res

    def create_or_restart_mt(self):
        lom_balancer = self._app_balancer_url('lom')
        mdb_balancer = self._app_balancer_url('mdb')

        self.Parameters.lom_mt_url = lom_balancer
        self.Parameters.mdb_mt_url = mdb_balancer

        try:
            logging.info("Starting MT")
            res = self._start_mt()
        except Exception as e:
            self._send_startrek_report("Ошибка при запуске среды: {}".format(e.message))
            raise e

        logging.info("TSUM response: {}".format(res.json()))

        launch_url = None
        if res is not None:
            try:
                launch_id = res.json()['launches'][-1]['number']
                launch_url = self._tsum_launch_url(self._tsum_page_url, launch_id)
            except Exception as e:
                logging.error("Can't find launch id in TSUM response")
                logging.error(e.message)

        if launch_url is not None:
            launch_or_environment_link = LAUNCH_LINK_MESSAGE.format(launch_url)
        else:
            launch_or_environment_link = ENVIRONMENT_LINK_MESSAGE.format(self._tsum_page_url)

        message = NOTIFICATION.format(launch_or_environment_link, lom_balancer, mdb_balancer)
        self._notify_pr(message)
        self._send_startrek_report(message)

    def on_execute(self):
        self.create_or_restart_mt()
