import logging
import os
import sandbox.common.types.client as ctc
import sandbox.common.types.misc as ctm
import tarfile
from sandbox import sdk2
from sandbox.sdk2.helpers import subprocess as sp


class MarketJavaSecMigration(sdk2.Task):
    class Requirements(sdk2.Task.Requirements):
        client_tags = ctc.Tag.Group.LINUX
        dns = ctm.DnsType.DNS64

    class Parameters(sdk2.Task.Parameters):
        app_resource = sdk2.parameters.Resource('app_resource')
        environment = sdk2.parameters.String('environment')
        domain = sdk2.parameters.String('domain')

    def on_execute(self):
        self.workdir = self.path('java-sec-migrator').as_posix()

        self.prepare_resource(self.Parameters.app_resource)

        args = ' '.join([
            os.path.join(self.workdir, 'bin', 'java-sec-migrator-start.sh'),
            '--environment={}'.format(self.Parameters.environment),
            "--extra-java-opts='-Ddomain={} -Dsandbox={}'".format(self.Parameters.domain, self.id)
        ])

        with sdk2.helpers.ProcessLog(self, logger=logging.getLogger("market-java-sec-migrator")) as pl:
            return_code = sp.Popen(args, shell=True, stdout=pl.stdout, stderr=sp.STDOUT, cwd=self.workdir).wait()
            if return_code != 0:
                raise Exception('Return code is not zero: ' + str(return_code))

    def prepare_resource(self, resource):
        resource = sdk2.ResourceData(resource)
        resource_path = resource.path.as_posix()

        logging.info('resource path: %s', resource_path)

        self.extract_with_strip(resource_path)

    def extract_with_strip(self, tar_path, strip=1):
        tar = tarfile.open(tar_path)
        for member in tar:
            split = member.name.split(os.path.sep)[strip:]
            if len(split) == 0:
                continue
            member.name = os.path.join(*split)
            tar.extract(member, path=self.workdir)
