import os

from sandbox import sdk2
from sandbox.sdk2.helpers import subprocess
import sandbox.common.types.client as ctc
from sandbox.common.errors import SubprocessError
from sandbox.projects.common.environments import SandboxJavaJdkEnvironment
from sandbox.projects.common.arcadia import sdk as arcadia_sdk

LIQUIBASE_RESOURCE_ID = 709114857
PG_JDBC_RESOURCE_ID = 709164335


"""
Copy task MarketLiquibaseUpdate
"""


class MarketLiquibaseUpdateMbo(sdk2.Task):
    """
    Task runs liquibase update.
    """

    class Requirements(sdk2.Task.Requirements):
        environments = [SandboxJavaJdkEnvironment('1.8.0')]
        client_tags = ctc.Tag.Group.LINUX

    class Parameters(sdk2.Task.Parameters):
        url = sdk2.parameters.String('Connection string', required=True)
        username = sdk2.parameters.String('Username', required=True)
        password_vault_name = sdk2.parameters.String('Password vault record name', required=True)
        password_vault_owner = sdk2.parameters.String('Password vault record owner', required=True)
        changelog_file = sdk2.parameters.String('Changelog file path', required=True)
        additional_options = sdk2.parameters.List('Additional options (e.g.: --delimiter=|)', default=[])
        with sdk2.parameters.String("Liquibase command", multiline=True) as command:
            command.values.updateSQL = command.Value(default=True)
            command.values.update = None
            command.values.rollbackCount = None
        jdbc_resource_id = sdk2.parameters.String('Jdbc driver resource id', required=False, default=PG_JDBC_RESOURCE_ID)
        liquibase_jar_resource_id = sdk2.parameters.String('Liquibase jar resource id', required=False, default=LIQUIBASE_RESOURCE_ID)
        additional_arguments = sdk2.parameters.List('Additional arguments', default=[])
        additional_java_properties = sdk2.parameters.List('Additional java properties', default=[])
        checkout_path = sdk2.parameters.String('Arcadia checkout path', required=True)
        arc_token_vault = sdk2.parameters.String('Arcadia token vault name', required=True)
        checkout_arcadia_from_url = sdk2.parameters.ArcadiaUrl(
            'Checkout arcadia from url',
            required=True
        )

    @staticmethod
    def get_resource_path(resource_id):
        return str(sdk2.ResourceData(sdk2.Resource.find(id=resource_id).first()).path)

    def on_prepare(self):
        os.environ["ARC_TOKEN"] = sdk2.Vault.data(self.Parameters.arc_token_vault)

    def on_execute(self):
        url = self.Parameters.url
        username = self.Parameters.username
        password = sdk2.Vault.data(self.Parameters.password_vault_owner, self.Parameters.password_vault_name)
        checkout_arcadia_from_url = self.Parameters.checkout_arcadia_from_url
        checkout_path = self.Parameters.checkout_path
        changelog_file = self.Parameters.changelog_file
        additional_options = self.Parameters.additional_options
        command = self.Parameters.command
        additional_arguments = self.Parameters.additional_arguments
        additional_java_properties = self.Parameters.additional_java_properties
        jdbc_resource_id = self.Parameters.jdbc_resource_id
        liquibase_jar_resource_id = self.Parameters.liquibase_jar_resource_id
        if not jdbc_resource_id:
            jdbc_resource_id = PG_JDBC_RESOURCE_ID
        if not liquibase_jar_resource_id:
            liquibase_jar_resource_id = LIQUIBASE_RESOURCE_ID

        liquibase_path = self.get_resource_path(liquibase_jar_resource_id)
        classpath = self.get_resource_path(jdbc_resource_id)
        with arcadia_sdk.mount_arc_path(url=checkout_arcadia_from_url, use_arc_instead_of_aapi=True) as arcadia_root:
            cwd = os.path.join(arcadia_root, checkout_path)
            with sdk2.helpers.ProcessLog(self, logger='liquibase') as pl:
                retcode = subprocess.Popen(
                    'java {additional_java_properties} -jar {liquibase} --classpath={classpath} --changeLogFile={changelog_file} '
                    '--url="{url}" --username={username} --password={password} {additional_options} {command} {additional_arguments}'.format(
                        additional_java_properties=' '.join(additional_java_properties),
                        liquibase=liquibase_path,
                        classpath=classpath,
                        changelog_file=changelog_file,
                        url=url,
                        username=username,
                        password=password,
                        additional_options=' '.join(additional_options),
                        command=command,
                        additional_arguments=' '.join(additional_arguments)
                    ),
                    shell=True, stdout=pl.stdout, stderr=subprocess.STDOUT, cwd=cwd
                ).wait()
                if retcode:
                    raise SubprocessError('Java process exited with non-zero return code {}'.format(retcode))
