import tempfile
from os import walk, path, mkdir

from sandbox import sdk2
from sandbox.sdk2.helpers import subprocess
import sandbox.common.types.client as ctc
from sandbox.common.errors import SubprocessError
from sandbox.projects.common.environments import SandboxJavaJdkEnvironment

LIQUIBASE_RESOURCE_ID = 709114857
PG_JDBC_RESOURCE_ID = 709164335


class MarketMboLiquibaseMigrationRun(sdk2.Task):
    """
    Clone of MarketLiquibaseUpdate adapted to github
    """

    class Requirements(sdk2.Task.Requirements):
        environments = [SandboxJavaJdkEnvironment('11.0.2')]
        client_tags = ctc.Tag.Group.LINUX

    class Parameters(sdk2.Task.Parameters):
        url = sdk2.parameters.String('Connection string', required=True)
        username = sdk2.parameters.String('Username', required=True)
        password_yav_key = sdk2.parameters.String('Password YaV key (format: sec-0abc@ver-0def[key] or sec-0abc[key])', required=True)
        changelog_jar_resource_id = sdk2.parameters.String('Changelog jar resource id', required=True)
        changelog_file = sdk2.parameters.String('Changelog file path', required=True)
        additional_options = sdk2.parameters.List('Additional options (e.g.: --delimiter=|)', default=[])
        with sdk2.parameters.String("Liquibase command", multiline=True) as command:
            command.values.updateSQL = command.Value(default=True)
            command.values.update = None
            command.values.rollbackCount = None
        jdbc_resource_id = sdk2.parameters.String('Jdbc driver resource id', required=False, default=PG_JDBC_RESOURCE_ID)
        liquibase_jar_resource_id = sdk2.parameters.String('Liquibase jar resource id', required=False, default=LIQUIBASE_RESOURCE_ID)
        additional_arguments = sdk2.parameters.List('Additional arguments', default=[])
        additional_java_properties = sdk2.parameters.List('Additional java properties', default=[])

    @staticmethod
    def get_resource_path(resource_id):
        return str(sdk2.ResourceData(sdk2.Resource.find(id=resource_id).first()).path)

    def on_execute(self):
        url = self.Parameters.url
        username = self.Parameters.username
        password = sdk2.Vault.data(self.Parameters.password_yav_key)
        changelog_jar_resource_id = self.Parameters.changelog_jar_resource_id
        changelog_file = self.Parameters.changelog_file
        additional_options = self.Parameters.additional_options
        command = self.Parameters.command
        additional_arguments = self.Parameters.additional_arguments
        additional_java_properties = self.Parameters.additional_java_properties
        jdbc_resource_id = self.Parameters.jdbc_resource_id
        liquibase_jar_resource_id = self.Parameters.liquibase_jar_resource_id
        if not jdbc_resource_id:
            jdbc_resource_id = PG_JDBC_RESOURCE_ID
        if not liquibase_jar_resource_id:
            liquibase_jar_resource_id = LIQUIBASE_RESOURCE_ID

        cwd = tempfile.mkdtemp()
        extracted_dir = path.join(cwd, "changelog_jar_extracted")
        mkdir(extracted_dir)

        changelog_jar_path = self.get_resource_path(changelog_jar_resource_id)
        with sdk2.helpers.ProcessLog(self, logger='tar') as pl:
            retcode = subprocess.Popen(
                "tar -xvf {changelog_jar_path} -C {extracted_dir}".format(changelog_jar_path=changelog_jar_path, extracted_dir=extracted_dir),
                shell=True,
                stdout=pl.stdout,
                stderr=subprocess.STDOUT,
                cwd=extracted_dir
            ).wait()
            if retcode:
                raise SubprocessError('tar process exited with non-zero return code {}'.format(retcode))

        jar_files = []
        for (dirpath, dirnames, filenames) in walk(extracted_dir):
            for file in filenames:
                jar_files += [path.join(dirpath, file)]

        liquibase_path = self.get_resource_path(liquibase_jar_resource_id)
        jdbc_path = self.get_resource_path(jdbc_resource_id)
        classpath_list = [jdbc_path] + jar_files
        classpath = ':'.join(classpath_list)
        with sdk2.helpers.ProcessLog(self, logger='liquibase') as pl:
            retcode = subprocess.Popen(
                'java {additional_java_properties} -jar {liquibase} --classpath={classpath} --changeLogFile={changelog_file} '
                '--url="{url}" --username={username} --password={password} {additional_options} {command} {additional_arguments}'.format(
                    additional_java_properties=' '.join(additional_java_properties),
                    liquibase=liquibase_path,
                    classpath=classpath,
                    changelog_file=changelog_file,
                    url=url,
                    username=username,
                    password=password,
                    additional_options=' '.join(additional_options),
                    command=command,
                    additional_arguments=' '.join(additional_arguments)
                ),
                shell=True, stdout=pl.stdout, stderr=subprocess.STDOUT, cwd=cwd
            ).wait()
            if retcode:
                raise SubprocessError('Java process exited with non-zero return code {}'.format(retcode))
