# coding: utf-8

import os
import shutil
import logging

from sandbox import sdk2
from sandbox.common.types import misc as ctm
from sandbox.sdk2.helpers import subprocess as sp
from sandbox.projects.common.arcadia.sdk import mount_arc_path
import sandbox.common.types.client as ctc


class MboYangGenerateTasksForThiriumE2ETests(sdk2.Task):
    work_dir_name = 'mbo-category-frontend'
    build_dir_name = 'build_dir'

    class Parameters(sdk2.Task.Parameters):
        description = "Task generates toloka tasks for thirium e2e tests"
        with sdk2.parameters.Group('Настройки репозитория') as repo_block:
            arc_path = sdk2.parameters.String(
                'Arcadia url:',
                required=True,
                default_value='arcadia-arc:/#trunk',
                description='Arc путь. EX: arcadia-arc:/#trunk или arcadia-arc:/#users/userName/brunch'
            )

            repo_path = sdk2.parameters.String(
                'Repo path:',
                required=True,
                description='Путь к репозиторию.',
                default_value='market/mbo/frontend/yang-react'
            )

        tokens = sdk2.parameters.YavSecret(
            "mbo-yang-tokens",
            default="sec-01g94qq70a9qh9px3b494hyaep"
        )

    class Requirements(sdk2.Requirements):
        dns = ctm.DnsType.DNS64
        ram = 2 * 1024
        disk_space = 2 * 1024
        client_tags = ctc.Tag.Group.LINUX & ctc.Tag.LINUX_XENIAL
        environments = (
            sdk2.environments.NodeJS('12.13.0', 'linux'),
        )

    def get_path_to_work_dir(self):
        """ Returns path to a working directory. """
        if self.work_dir_name is None:
            raise AttributeError('Name of working directory is not defined')

        return str(self.path(self.work_dir_name))

    def checkout_arcadia_repository(self):
        logging.debug('Checkout arcadia repo {}'.format(self.Parameters.repo_path))
        with mount_arc_path(self.Parameters.arc_path, use_arc_instead_of_aapi=True) as arcadia:
            self.work_dir_name = os.path.join(arcadia, self.Parameters.repo_path)

            self.build_dir_name = os.path.join(str(self.path('')).rstrip('/'), 'build_dir')
            logging.debug('Copy arcadia to local build dir > {} to {}'.format(self.work_dir_name,
                                                                              self.build_dir_name))
            shutil.copytree(self.work_dir_name, self.build_dir_name)

            self.work_dir_name = self.build_dir_name

    def prepare_node_modules(self):
        path_to_work_dir = self.get_path_to_work_dir()

        with sdk2.helpers.ProcessLog(self, logger='preparation') as plp:
            sp.check_call(['npm', 'ci', '--registry=https://npm.yandex-team.ru'], cwd=path_to_work_dir,
                          stdout=plp.stdout, stderr=plp.stderr)

    def generateTasks(self):
        path_to_work_dir = self.get_path_to_work_dir()

        oauth_token = self.Parameters.tokens.data()['toloka-req-oauth-token']

        with sdk2.helpers.ProcessLog(self, logger='build') as pl:
            sp.check_call(['npm', 'run', 'generateTolokaTasks'], cwd=path_to_work_dir,
                          stdout=pl.stdout, stderr=pl.stderr,
                          env={'TOLOKA_OAUTH_TOKEN': oauth_token, 'TOLOKA_PROJECTS': [39792, 40128]})

    def on_execute(self):
        self.checkout_arcadia_repository()
        self.prepare_node_modules()
        self.generateTasks()
