# -*- coding: utf-8 -*-
import json
import logging
from datetime import datetime

from sandbox.projects.common import binary_task

from sandbox import sdk2
from sandbox.projects.market.mcrp.MarketMcrpAbcSync.abc import Dispenser, parse_changes, RequestValueError

# CSADMIN-45125
YT_ACL = [
    {
        "subjects": ["idm-group:41953"],  # Группа на хане, аналог svc_marketito_administration
        "permissions": ["read", "manage"],
        "action": "allow",
    }
]

REQUESTS_SCHEMA = [
    {"name": "id", "type": "uint64", "sort_order": "ascending", "required": True},
    {"name": "abcID", "type": "uint64", "required": True},
    {"name": "created", "type": "uint64", "required": True},
    {"name": "updated", "type": "uint64"},
    {"name": "abcTicket", "type": "string"},
    {"name": "preorderId", "type": "string"},
    {"name": "abc", "type": "string", "required": True},
    {"name": "summary", "type": "string"},
    {"name": "description", "type": "string"},
    {"name": "requestType", "type": "string"},
    {"name": "createdBy", "type": "string", "required": True},
    {"name": "responsible", "type": "string"},
    {"name": "resources", "type": "any"},
    {"name": "resourcesReady", "type": "any"},
    {"name": "resourcesAllocated", "type": "any"},
    {"name": "status", "type": "string", "required": True},
    {"name": "comment", "type": "string"},
    {"name": "calculations", "type": "string"},
    {"name": "chartLinks", "type": "any"},
    {"name": "requestGoalAnswers", "type": "any"},
    {"name": "goal", "type": "string"},
    {"name": "goalId", "type": "uint64"},
    {"name": "goalName", "type": "string"},
]

COSTS_SCHEMA = [
    {"name": "abcTicket", "type": "string", "sort_order": "ascending", "required": True},
    {"name": "cost", "type": "double", "required": True},
]


class MarketMcrpAbcSync(binary_task.LastBinaryTaskRelease, sdk2.Task):
    """MCRP Sync ABC Tasks"""

    class Parameters(sdk2.Task.Parameters):
        # common parameters
        kill_timeout = 3600

        # custom parameters
        yt_cluster = sdk2.parameters.String("YT cluster", default="hahn", required=True)
        abc_request_db_path = sdk2.parameters.String("MCRP ABC Requests YT db directory path", required=True)
        money_db_path = sdk2.parameters.String("MCRP Money db path", required=True)
        yt_token = sdk2.parameters.YavSecret("yav with yt token as 'yt' key", required=True)
        dispenser_token = sdk2.parameters.YavSecret("yav with dispenser token as 'dispenser' key", required=True)
        order_name = sdk2.parameters.String("ABC Order name", required=True)
        parent_abc = sdk2.parameters.Integer("Parent ABC service ID", required=True)

        ext_params = binary_task.binary_release_parameters(stable=True)

    @staticmethod
    def get_db_path(db_path):
        return str(db_path).rstrip("/")

    @staticmethod
    def update_insert_row(sequence, yt_client, abc_table, costs_table, abc_request):
        for row in yt_client.select_rows(
            'abcID, updated from [{}] where abcID = {}'.format(abc_table, abc_request['id'])
        ):
            logging.info(row)
            if abc_request.get('trackerIssueKey'):
                update_cost = False
                for cost_row in yt_client.select_rows(
                    'abcTicket, cost from [{}] where abcTicket = "{}"'.format(costs_table, abc_request['trackerIssueKey'])
                ):
                    if cost_row['cost'] != abc_request.get('cost', 0):
                        update_cost = True
                if update_cost:
                    yt_client.insert_rows(costs_table, [{
                        'abcTicket': abc_request['trackerIssueKey'],
                        'cost': abc_request.get('cost', 0),
                    }])
            if abc_request['updated'] <= row['updated']:
                return False
        try:
            goal = abc_request['goal'] if abc_request.get('goal') else {}
            yt_client.insert_rows(abc_table, [{
                'id': sequence,
                'abcID': abc_request['id'],
                'created': abc_request['created'],
                'updated': abc_request['updated'],
                'abcTicket': abc_request.get('trackerIssueKey'),
                'preorderId': abc_request['campaign']['key'],
                'requestType': abc_request.get('resourcePreorderReasonType'),
                'abc': abc_request['project']['key'].lower(),
                'summary': abc_request.get('summary'),
                'createdBy': abc_request['author'],
                'responsible': abc_request.get('responsible'),
                'status': abc_request['status'],
                'comment': abc_request.get('comment'),
                'description': abc_request.get('description'),
                'calculations': abc_request.get('calculations'),
                'chartLinks': abc_request.get('chartLinks'),
                'requestGoalAnswers': abc_request.get('requestGoalAnswers'),
                'resources': parse_changes(abc_request['changes']),
                'resourcesReady': parse_changes(abc_request['changes'], 'amountReady'),
                'resourcesAllocated': parse_changes(abc_request['changes'], 'amountAllocated'),
                'goal': 'https://goals.yandex-team.ru/filter?goal={}'.format(goal.get('id')),
                'goalId': goal.get('id'),
                'goalName': goal.get('name'),
            }])
        except RequestValueError:
            logging.error(json.dumps(abc_request, indent=2))
            raise
        return True

    def on_execute(self):
        import yt.wrapper as yt
        yt_client = yt.YtClient(proxy=self.Parameters.yt_cluster,
                                token=self.Parameters.yt_token.data()["yt"],
                                config={"backend": "rpc"})
        yt_client.config['spec_overrides'] = {'acl': YT_ACL}
        # yt_format = yt.JsonFormat()
        abc_path = yt.YPath(self.get_db_path(self.Parameters.abc_request_db_path))
        if not yt_client.exists(abc_path):
            yt_client.mkdir(abc_path)

        money_path = yt.YPath(self.get_db_path(self.Parameters.money_db_path))
        if not yt_client.exists(money_path):
            yt_client.mkdir(money_path)

        abc_table = yt.TablePath(abc_path.join('requests'))
        costs_table = yt.TablePath(money_path.join('costs'))
        sequence_path = yt.YPath(abc_path.join('sequence'))
        synced_path = yt.YPath(abc_path.join('synced'))
        broken_path = yt.YPath(abc_path.join('broken'))
        yt_client.create('int64_node', sequence_path, ignore_existing=True)

        yt_client.create('table', abc_table, ignore_existing=True, attributes={
            'dynamic': True,
            'schema': REQUESTS_SCHEMA,
            'enable_dynamic_store_read': True
        })
        yt_client.create('table', costs_table, ignore_existing=True, attributes={
            'dynamic': True,
            'schema': COSTS_SCHEMA,
            'enable_dynamic_store_read': True
        })
        yt_client.mount_table(abc_table, sync=True)
        yt_client.mount_table(costs_table, sync=True)

        yt_lock_client = yt.YtClient(config=yt.config.get_config(yt_client))

        try:
            with Dispenser(self.Parameters.dispenser_token.data()["dispenser"]) as dispenser:
                campaign_orders = []
                for campaign_order in dispenser.get_campaign_orders():
                    if campaign_order['name'].lower() == self.Parameters.order_name.lower():
                        for big_order in campaign_order['campaignBigOrders']:
                            campaign_orders.append(big_order['id'])

                # transaction_id = yt_client.start_transaction(type='master', sticky=True)
                with yt_lock_client.Transaction():
                    if not yt_lock_client.lock(sequence_path):
                        return
                    with yt_client.Transaction(type="tablet"):
                        sequence = yt_lock_client.get(sequence_path)
                        for abc_request in dispenser.get_quota_requests(self.Parameters.parent_abc, campaign_orders):
                            if self.update_insert_row(sequence, yt_client, abc_table, costs_table, abc_request):
                                sequence += 1
                        yt_lock_client.set(sequence_path, sequence)
                    yt_lock_client.set(synced_path, int(datetime.utcnow().strftime('%s')))
                    yt_lock_client.set(broken_path, 0)
        except RequestValueError:
            yt_lock_client.set(broken_path, 1)
            raise

