import requests
from enum import Enum
from collections import defaultdict
import os


class RequestValueError(ValueError):
    pass


class Dispenser(requests.Session):
    API_URL = 'https://dispenser.yandex-team.ru/common/api/'
    BASE_URL = API_URL + 'v1/'

    def __init__(self, token=None):
        super(Dispenser, self).__init__()
        if token is None:
            if 'DISPENSER_TOKEN' in os.environ:
                token = os.environ['DISPENSER_TOKEN']
            elif 'DISPENSER_TOKEN_PATH' in os.environ:
                with open(os.path.expanduser(os.environ['DISPENSER_TOKEN_PATH'])) as f:
                    token = f.read().strip()
            else:
                raise RuntimeError('Dispenser token is required')
        self.headers.update({
            'Accept': 'application/json;charset=UTF-8',
            'Content-Type': 'application/json;charset=UTF-8',
            'Authorization': 'OAuth {}'.format(token)
        })

    def get_quota_requests(self, ancestor_abc_service_id, campaign_orders):
        params = [('ancestorAbcServiceId', ancestor_abc_service_id), ('abcServiceId', ancestor_abc_service_id)]
        for campaign in campaign_orders:
            params.append(('campaignOrder', campaign))
        res = self.get(self.BASE_URL + 'quota-requests', params=params)
        res.raise_for_status()

        return res.json()['result']

    def get_campaign_orders(self):
        res = self.get(self.BASE_URL + 'campaigns')
        res.raise_for_status()

        return res.json()['result']

    def get_dictionaries(self):
        res = self.get(self.API_URL + 'front/dictionaries', params={'dictionaries': 'CAMPAIGN_PROVIDERS_SETTINGS'})
        res.raise_for_status()

        return res.json()['CAMPAIGN_PROVIDERS_SETTINGS']


class ABCEntity(Enum):
    @classmethod
    def from_str(cls, location, raise_error=True):
        location_key = location.lower()
        for name, member in cls.__members__.items():
            if name.lower() == location_key or location_key in member.keys:
                return member

        if raise_error:
            raise RequestValueError('Invalid {}: "{}"'.format(cls.__name__, location))
        return None

    @classmethod
    def from_iterator(cls, iterator):
        for item in iterator:
            value = cls.from_str(item, raise_error=False)
            if value is not None:
                return value
        raise RequestValueError('{} not found in iterator'.format(cls.__name__))

    @property
    def key(self):
        return self.value[0].lower()

    @property
    def keys(self):
        return {key.lower() for key in self.value}

    def __str__(self):
        return self.value[0]

    def __gt__(self, other):
        return self.name.lower().__gt__(other.name.lower())

    def __lt__(self, other):
        return self.name.lower().__lt__(other.name.lower())

    def __eq__(self, other):
        return self.name.lower().__eq__(other.name.lower())

    def __hash__(self):
        return self.name.lower().__hash__()


class Cloud(ABCEntity):
    MDB = ('MDB', 'dbaas')
    MDS = ('MDS',)
    LogBrokerYTProxy = ('LogBrokerYTProxy', 'logbroker_yt_proxy')
    YT = ('YT',)
    YP = ('YP',)
    RTC = ('RTC', 'gencfg')
    Sandbox = ('Sandbox',)
    LogBroker = ('LogBroker',)
    Nirvana = ('Nirvana',)
    SAAS = ('SAAS',)
    SQS = ('SQS',)
    YDB = ('YDB',)
    DistBuild = ('DistBuild',)
    RTMRProcessing = ('RTMRProcessing', 'rtmr_processing')
    RTMRMirror = ('RTMRMirror', 'rtmr_mirror')
    RTMR = ('RTMR',)
    Logfeller = ('Logfeller',)
    Solomon = ('Solomon',)


class MDBCloud(ABCEntity):
    POSTGRESQL = ('PostgreSQL', 'pgsql', 'dbaas_pgsql')
    CLICKHOUSE = ('ClickHouse', 'dbaas_clickhouse')
    MONGODB = ('MongoDB', 'dbaas_mongodb')
    REDIS = ('Redis', 'dbaas_redis')
    MYSQL = ('MySQL', 'dbaas_mysql')
    KAFKA = ('Kafka', 'dbaas_kafka')
    ELASTICSEARCH = ('Elasticsearch', 'dbaas_elasticsearch')
    GREENPLUM = ('Greenplum', 'dbaas_greenplum')


class DC(ABCEntity):
    SAS, VLA, MAN, IVA, MYT = ('SAS',), ('VLA',), ('MAN',), ('IVA',), ('MYT',)


class YTCluster(ABCEntity):
    HAHN = ('HAHN', 'Hahn')
    ARNOLD = ('ARNOLD', 'Arnold')
    SENECA_SAS = ('SENECA_SAS', 'Seneca-SAS')
    SENECA_VLA = ('SENECA_VLA', 'Seneca-VLA')
    SENECA_MAN = ('SENECA_MAN', 'Seneca-MAN')
    MARKOV = ('MARKOV', 'Markov')
    PYTHIA = ('PYTHIA', 'Pythia')
    ZENO = ('ZENO', 'Zeno')
    FREUD = ('FREUD', 'freud')
    HUME = ('HUME', 'hume')
    LANDAU = ('LANDAU', 'landau')
    BOHR = ('BOHR', 'bohr')
    LOCKE = ('LOCKE', 'locke')
    VANGA = ('VANGA', 'vanga')


class LogBrokerCluster(ABCEntity):
    SAS = ('SAS', 'logbroker_SAS')
    VLA = ('VLA', 'logbroker_VLA')
    MAN = ('MAN', 'logbroker_MAN')
    IVA = ('IVA', 'logbroker_IVA')
    MYT = ('MYT', 'logbroker_MYT')
    LBKX = ('LBKX',)
    PRESTABLE_SAS = ('PrestableSAS', 'logbroker-prestable_SAS')
    PRESTABLE_VLA = ('PrestableVLA', 'logbroker-prestable_VLA')
    PRESTABLE_MAN = ('PrestableMAN', 'logbroker-prestable_MAN')
    PRESTABLE_IVA = ('PrestableIVA', 'logbroker-prestable_IVA')
    PRESTABLE_MYT = ('PrestableMYT', 'logbroker-prestable_MYT')
    YC = ('YC', 'logbroker-yc')
    PREPROD_YC = ('PreprodYC', 'logbroker-yc-preprod')


class Resource(ABCEntity):
    CPUCores = ('CPUCores', 'cpu', 'cpu_segmented', 'userpool_cores', 'cpu_linux', 'burst_guarantee_cpu', 'cpu_flow')
    RAMGb = ('RAMGb', 'ram', 'ram_segmented', 'ram_linux', 'mem_strong', 'mem_burst', 'mem_relaxed', 'ram_gig')
    HDDGb = ('HDDGb', 'hdd', 'hdd_segmented', 'hdd_linux')
    SSDGb = ('SSDGb', 'ssd', 'ssd_segmented', 'ssd_linux')
    StorageGb = ('StorageGb', 'ydb_ru-data_size', 's3-api', 'avatars', 'data_size', 'mds', 'storage-s3', 's3-storage',
                 's3_storage')
    DataGb = ('DataGb', 'data')
    RPS = ('RPS', 'ydb_ru-rps')
    KPS = ('KPS',)
    GPUCards = ('GPUCards', 'gpu', 'gpu_segmented')
    IOSSDBps = ('IOSSDBps', 'io_ssd')
    IOHDDBps = ('IOHDDBps', 'io_hdd')
    DinTableNodes = ('DinTableNodes', 'tablet_cell_bundle')
    DinTableRAMGb = ('DinTableRAMGb', 'tablet_static_memory')
    CHYTCPUCores = ('CHYTCPUCores', 'cpu_clickhouse')
    NetworkBps = ('NetworkBps', 'cluster_network_traffic', 'strm_outbound_external_traffic')
    Segment = ('Segment',)
    DataFlowBps = ('DataFlowBps', 'data_flow', 'dataflow')
    WriteThroughput = ('WriteThroughput', 'write_capacity')
    WriteThroughputBinary = ('WriteThroughputBinary', 'data_flow_binary')
    MetricsFlow = ('MetricsFlow', 'metrics_write_flow', 'metrics_write_flow_segmented')
    MetricsStored = ('MetricsStored', 'metrics_stored_count')
    RPCProxy = ('RPCProxy', 'rpc_proxy')
    SBMacMini = ('SBMacMini', 'mac_mini')
    SBWindows = ('SBWindows', 'windows')
    YFSlots = ('YFSlots', 'yf_slot')
    STRMLiveCasts = ('STRMLiveCasts', 'strm_simultaneous_live_casts')


class SandboxSegment(ABCEntity):
    LinuxYP = ('LinuxYP', 'linux_yp', 'sandbox_linux_yp')
    Linux = ('Linux', 'linux', 'sandbox_linux', 'cpu_linux', 'ram_linux', 'ssd_linux', 'hdd_linux', 's3_storage')
    LinuxBareMetal = ('LinuxBareMetal', 'sandbox_linux_bare_metal')
    MacMini = ('mac_mini',)
    Windows = ('windows',)


class DistBuildSegment(ABCEntity):
    User = ('User', 'distbuild_user')
    AutoCheck = ('AutoCheck', 'distbuild_autocheck')


class MDSType(ABCEntity):
    MDS = ('MDS',)
    S3 = ('S3', 's3-api')
    Avatars = ('Avatars',)


class Change(object):
    def __init__(self, cloud, date, location, resource, value):
        self.cloud = cloud
        self.date = date
        self.location = location
        self.resource = resource
        self.value = value


def get_value(value):
    if value['unit'] == 'BYTE':
        return value['value'] / 1024 / 1024 / 1024
    elif value['unit'] == 'PERMILLE_CORES':
        return value['value'] / 1000
    elif value['unit'] == 'PERMILLE':
        return value['value'] / 1000
    elif value['unit'] == 'BPS':
        return value['value']
    elif value['unit'] == 'BINARY_BPS':
        return value['value']
    raise RequestValueError('Unknown value')


def parse_change(change, tp='amount'):
    cloud = Cloud.from_str(change['service']['key'])
    if cloud in {Cloud.LogBrokerYTProxy, Cloud.YT, Cloud.Logfeller}:
        return Change(cloud,
                      change['order']['orderDate'],
                      YTCluster.from_iterator(change['segmentKeys']),
                      Resource.from_str(change['resource']['key']),
                      get_value(change[tp]))
    elif cloud in {Cloud.YP, Cloud.Nirvana, Cloud.SAAS, Cloud.RTC}:
        return Change(cloud,
                      change['order']['orderDate'],
                      DC.from_iterator(change['segmentKeys']) if change['segmentKeys'] else 'Default',
                      Resource.from_str(change['resource']['key']),
                      get_value(change[tp]))
    elif cloud == Cloud.MDB:
        return Change(cloud,
                      change['order']['orderDate'],
                      MDBCloud.from_iterator(change['segmentKeys']),
                      Resource.from_str(change['resource']['key']),
                      get_value(change[tp]))
    elif cloud == Cloud.LogBroker:
        return Change(cloud,
                      change['order']['orderDate'],
                      LogBrokerCluster.from_iterator(change['segmentKeys']),
                      Resource.from_str(change['resource']['key']),
                      get_value(change[tp]))
    elif cloud == Cloud.Sandbox:
        return Change(cloud,
                      change['order']['orderDate'],
                      SandboxSegment.from_iterator(change['segmentKeys'] + [change['resource']['key']]),
                      Resource.from_str(change['resource']['key']),
                      get_value(change[tp]))
    elif cloud in {Cloud.SQS, Cloud.YDB, Cloud.RTMRMirror, Cloud.RTMRProcessing, Cloud.RTMR, Cloud.Solomon}:
        return Change(cloud,
                      change['order']['orderDate'],
                      'Default',
                      Resource.from_str(change['resource']['key']),
                      get_value(change[tp]))
    elif cloud == Cloud.MDS:
        return Change(cloud,
                      change['order']['orderDate'],
                      MDSType.from_str(change['resource']['key']),
                      Resource.from_str(change['resource']['key']),
                      get_value(change[tp]))
    elif cloud == Cloud.DistBuild:
        return Change(cloud,
                      change['order']['orderDate'],
                      DistBuildSegment.from_iterator(change['segmentKeys']),
                      Resource.from_str(change['resource']['key']),
                      get_value(change[tp]))


def parse_changes(changes, tp='amount'):
    result = defaultdict(lambda: defaultdict(lambda: defaultdict(lambda: [defaultdict(int)])))
    for change in changes:
        c = parse_change(change, tp)
        result[str(c.cloud)][str(c.date)][str(c.location)][0][str(c.resource)] += c.value

    return result


