# -*- coding: utf-8 -*-
import logging
import json
from sandbox.projects.common import binary_task
from collections import defaultdict
from sandbox.projects.market.mcrp.MarketMcrpAbcTopServicesSync.abc import Abc

from sandbox import sdk2

# CSADMIN-45125
YT_ACL = [
    {
        "subjects": ["idm-group:41953"],  # Группа на хане, аналог svc_marketito_administration
        "permissions": ["read", "manage"],
        "action": "allow",
    }
]

MAP_SCHEMA = [
    {"name": "abc", "type": "string", "sort_order": "ascending", "required": True},
    {"name": "abc_name", "type": "string", "required": True},
    {"name": "children", "type_v3": {"type_name": "list", "item": "string"}, "required": True},
    {"name": "children_names", "type_v3": {"type_name": "list", "item": "string"}, "required": True},
]


class MarketMcrpAbcTopServicesSync(binary_task.LastBinaryTaskRelease, sdk2.Task):
    """MCRP Sync ABC Top Services"""

    class Parameters(sdk2.Task.Parameters):
        # common parameters
        kill_timeout = 3600

        # custom parameters
        yt_cluster = sdk2.parameters.String("YT cluster", default="hahn", required=True)
        abc_map_db_path = sdk2.parameters.String("MCRP ABC Map YT db directory path", required=True)
        yt_token = sdk2.parameters.YavSecret("yav with yt token as 'yt' key", required=True)
        abc_token = sdk2.parameters.YavSecret("yav with abc token as 'abc' key", required=True)

        ext_params = binary_task.binary_release_parameters(stable=True)

    @staticmethod
    def get_db_path(db_path):
        return str(db_path).rstrip("/")

    def on_execute(self):
        import yt.wrapper as yt
        yt_client = yt.YtClient(proxy=self.Parameters.yt_cluster,
                                token=self.Parameters.yt_token.data()["yt"])
        yt_client.config['spec_overrides'] = {'acl': YT_ACL}
        # yt_format = yt.JsonFormat()
        abc_path = yt.YPath(self.get_db_path(self.Parameters.abc_map_db_path))
        if not yt_client.exists(abc_path):
            yt_client.mkdir(abc_path)

        top_services_map_table = yt.TablePath(abc_path.join('top-services-map'), schema=MAP_SCHEMA)
        top_services_path = yt.YPath(abc_path.join('top-services'))

        top_services = set(yt_client.get(top_services_path))

        with Abc(self.Parameters.abc_token.data()["abc"]) as abc_client:
            services = abc_client.get_service_by_slug_with_descendants('meta_market')

            parents = defaultdict(list)
            abc_names = dict()
            result = defaultdict(list)

            for service in services:
                parents[service['parent']['slug'].lower()].append(service['slug'].lower())
                if service['slug'].lower() not in abc_names.keys():
                    abc_names[service['slug'].lower()] = service['name']['ru']
                if service['parent']['slug'].lower() not in abc_names.keys():
                    abc_names[service['parent']['slug'].lower()] = service['parent']['name']['ru']
                # print(json.dumps(service, indent=2))

            top_services.update(parents['meta_market'])
            top_services.update(parents['market_map'])

            def get_children(root_service, parent):
                for name in parents[parent]:
                    if name in top_services:
                        result[name] = []
                        get_children(name, name)
                    else:
                        result[root_service].append(name)
                        get_children(root_service, name)

            result['meta_market'] = []
            get_children('meta_market', 'meta_market')

            yt_client.write_table(top_services_map_table,
                                  [{"abc": i, "abc_name": abc_names[i], "children": j, 
                                    "children_names": [abc_names[x] for x in j]} for i, j in sorted(result.items())])
