# -*- coding: utf-8 -*-
import json
import logging
from datetime import datetime

from sandbox.projects.common import binary_task

from sandbox import sdk2

# CSADMIN-45125
YT_ACL = [
    {
        "subjects": ["idm-group:41953"],  # Группа на хане, аналог svc_marketito_administration
        "permissions": ["read", "manage"],
        "action": "allow",
    }
]

TIMESTAMP_MASK = ((1 << 32) - 1)
COUNTER_BITS = 30

REPORT_SCHEMA = [
    {"name": "ticket", "type": "string", "sort_order": "ascending", "required": True},
    {"name": "id", "type": "uint64"},
    {"name": "created", "type": "uint64"},
    {"name": "updated", "type": "uint64"},
    {"name": "mcrpTicket", "type": "string"},
    {"name": "abcTicket", "type": "string"},
    {"name": "preorderId", "type": "string"},
    {"name": "requestType", "type": "string"},
    {"name": "version", "type": "string"},
    {"name": "name", "type": "string"},
    {"name": "abc", "type": "string"},
    {"name": "abcName", "type": "string"},
    {"name": "topAbc", "type": "string"},
    {"name": "topAbcName", "type": "string"},
    {"name": "cloud", "type": "string"},
    {"name": "clouds", "type": "string"},
    {"name": "platform", "type": "string"},
    {"name": "when", "type": "string"},
    {"name": "reason", "type": "string"},
    {"name": "createdBy", "type": "string"},
    {"name": "approvedBy", "type": "string"},
    {"name": "resources", "type": "string"},
    {"name": "status", "type": "string"},
    {"name": "generatedBy", "type": "string"},
    {"name": "sourceMeta", "type": "string"},
    {"name": "sourceId", "type": "string"},
    {"name": "goal", "type": "string"},
    {"name": "goalId", "type": "int64"},
    {"name": "goalName", "type": "string"},
    {"name": "resps", "type_v3": {"type_name": "list", "item": "string"}},
    {"name": "whatifno", "type": "string"},
    {"name": "deadline", "type": "string"},
    {"name": "projectTicket", "type": "string"},
    {"name": "impactOn", "type": "string"},
    {"name": "impactOnKPI", "type": "string"},
    {"name": "limit100p", "type": "string"},
    {"name": "limit50p", "type": "string"},
    {"name": "priority", "type": "string"},
]

UPDATED_SCHEMA = [
    {"name": "type", "type": "string", "sort_order": "ascending", "required": True},
    {"name": "when", "type": "uint64"},
]

COSTS_SCHEMA = [
    {"name": "abcTicket", "type": "string", "sort_order": "ascending", "required": True},
    {"name": "cost", "type": "double", "required": True},
]


class TaskError(RuntimeError):
    pass


def get_timestamp_from_id(request_id):
    """30-61 bits of id"""
    return (request_id >> COUNTER_BITS) & TIMESTAMP_MASK


class RequestReducer(object):
    def __init__(self, abc_mapping, goals_mapping, valid_keys):
        self.abc_mapping = abc_mapping
        self.goals_mapping = goals_mapping
        self.valid_keys = valid_keys

    def get_enrich_abc(self, abc):
        defaultResponse = {'topAbc': abc,
                           'topAbcName': abc,
                           'abcName': abc}
        abc = abc.lower()
        return self.abc_mapping.get(abc, defaultResponse)


class McrpReducer(RequestReducer):
    mdb_clouds = {
        'MONGO': 'MongoDB',
        'PGAAS': 'PostgreSQL',
        'CLICKHOUSE': 'ClickHouse',
        'MYSQL': 'MySQL'
    }

    sanitize_kind = {
        'yt': 'YT',
        'logbrokerytproxy': 'LogBrokerYTProxy',
        'logfeller': 'Logfeller',
        'yp': 'YP',
        'rtc': 'RTC',
        'saas': 'SAAS',
        'nirvana': 'Nirvana',
        'mdb': 'MDB',
        'mds': 'MDS',
        'logbroker': 'LogBroker',
        'sandbox': 'Sandbox',
        'sqs': 'SQS',
        'ydb': 'YDB',
        'rtmrprocessing': 'RTMRProcessing',
        'rtmrmirror': 'RTMRMirror',
        'solomon': 'Solomon',
        'distbuild': 'DistBuild',
        'baremetal': 'BareMetal',

    }
    sanitize_locations = {
        'sas': 'SAS',
        'vla': 'VLA',
        'man': 'MAN',
        'iva': 'IVA',
        'myt': 'MYT',
        'prestablesas': 'PrestableSAS',
        'prestablevla': 'PrestableVLA',
        'prestableman': 'PrestableMAN',
        'prestableiva': 'PrestableIVA',
        'prestablemyt': 'PrestableMYT',
        'lbkx': 'LBKX',
        'hahn': 'HAHN',
        'arnold': 'ARNOLD',
        'seneca-sas': 'SENECA_SAS',
        'seneca-vla': 'SENECA_VLA',
        'seneca-man': 'SENECA_MAN',
        'markov': 'MARKOV',
        'pythia': 'PYTHIA',
        'zeno': 'ZENO',
        'postgresql': 'PostgreSQL',
        'clickhouse': 'ClickHouse',
        'mongodb': 'MongoDB',
        'redis': 'Redis',
        'mysql': 'MySQL',
        'kafka': 'Kafka',
        'elasticsearch': 'Elasticsearch',
        'greenplum': 'Greenplum',
        's3': 'S3',
        'avatars': 'Avatars',
        'mds': 'MDS',
        'linuxyp': 'LinuxYP',
        'linux': 'Linux',
        'linusbaremetal': 'LinusBareMetal',
        'default': 'Default',
        'autocheck': 'AutoCheck',
        'user': 'User',
        'warehouse': 'WareHouse',
        'macmini': 'MacMini',
        'windows': 'Windows'
    }

    def fix_clouds(self, cloud, resources):
        updates_resources = {}
        if cloud == 'MDB':
            updates_resources[self.mdb_clouds.get(resources['Cloud'], resources['Cloud'])] = [resources]
        elif cloud in {'MDS', 'Avatars', 'S3'}:
            updates_resources[cloud] = [resources]
        elif cloud == 'Sandbox':
            if resources.get('Segment', 'linux_yp') == 'linux_yp':
                updates_resources['LinuxYP'] = [resources]
            else:
                updates_resources[resources['Segment']] = [resources]
        elif cloud == 'YDB':
            updates_resources['Default'] = [resources]
        elif 'locations' in resources:
            updates_resources = resources['locations']
        else:
            return resources
        return updates_resources

    @staticmethod
    def fix_priority(priority):
        if priority is None:
            priority = ''
        priority = priority.lower()
        if priority in {'critical', u'критичный'}:
            return 'critical'
        elif priority in {'blocker', u'блокер'}:
            return 'blocker'
        return 'normal'

    def __call__(self, key, rows):
        try:
            last_row = None
            created = None

            for row in rows:
                if row["ticket"] is None or row["ticket"] == "":
                    return
                if created is None:
                    created = get_timestamp_from_id(row["id"])
                last_row = row

            if last_row is not None:
                row = {key: last_row.get(key) for key in self.valid_keys}
                try:
                    source_meta = json.loads(last_row["sourceMeta"])
                except ValueError as exc:
                    logging.error('{}: {}'.format(exc.__class__.__name__, exc))
                    source_meta = {}
                migrated = source_meta.get("migrated", False)
                resources = json.loads(last_row["resources"])
                row["created"] = created * 1000
                row["updated"] = get_timestamp_from_id(last_row["id"]) * 1000
                row["version"] = "main"
                row["mcrpTicket"] = last_row["ticket"]
                row["abc"] = last_row["abc"].lower()
                row["topAbc"] = self.get_enrich_abc(last_row["abc"]).get('topAbc')
                row["abcName"] = self.get_enrich_abc(last_row["abc"]).get('abcName')
                row["topAbcName"] = self.get_enrich_abc(last_row["abc"]).get('topAbcName')
                # migrated по сути v2 mcrp.
                if migrated:
                    # Могут быть разночтения в названиях облаков и локаций, приходится приводить
                    row["resources"] = {k: {
                        self.sanitize_locations.get(i.lower(), i): j
                        for i, j in v['locations'].items()
                    }
                        for k, v in resources.items()
                    }
                    row['cloud'] = self.sanitize_kind.get(str(last_row['cloud']).lower(), last_row['cloud'])
                else:
                    row["resources"] = {
                        row["deadline"]: self.fix_clouds(last_row["cloud"], json.loads(last_row["resources"]))
                    }
                if migrated:
                    row["when"] = [k for k in resources.keys()]
                else:
                    row["when"] = [last_row["deadline"]]
                row["requestType"] = "GROWTH"
                if row["abcTicket"]:
                    row["ticket"] = last_row["abcTicket"]
                try:
                    if "version" in source_meta:
                        row["version"] = source_meta["version"]
                    if "goal" in source_meta:
                        goal = source_meta["goal"]
                        row["goal"] = goal
                        goalDetails = self.goals_mapping.get(goal)
                        if hasattr(goalDetails, 'get'):
                            row['goalId'] = int(goalDetails.get('goalId'))
                            row['goalName'] = goalDetails.get('goalName')
                    else:
                        row['goalId'] = -1
                    for s, k in [
                        ('request_type', 'request_type'),
                        ('requestType', 'requestType'),
                        ('abcTicket', 'abcTicket'),
                        ('project', 'projectTicket'),
                        ('impactOn', 'impactOn'),
                        ('impactOnKPI', 'impactOnKPI'),
                        ('limit100p', 'limit100p'),
                        ('limit50p', 'limit50p'),
                        ('priority', 'priority'),
                    ]:
                        if s in source_meta:
                            row[k] = source_meta[s]

                    row['priority'] = self.fix_priority(row.get('priority', ''))

                except ValueError:
                    pass
                yield row
        except Exception as exc:
            raise TaskError("{}: {}: {}".format(key, exc.__class__.__name__, exc))


class AbcReducer(RequestReducer):
    @staticmethod
    def parse_preorder_id(preorder_id):
        date = datetime.strptime(preorder_id, '%b%Y')
        if date.month < 7:
            order_id = 'AO'
            year = date.year
        else:
            order_id = 'PO'
            year = date.year + 1
        return '{}_{}_{:%Y%m}'.format(order_id, year, date)

    @staticmethod
    def get_deadline(deadline):
        parts = deadline.split('-')
        if len(parts) < 2:
            return deadline
        return '-'.join(parts[0:2])

    def get_reasons(self, row, request_goal_answers):
        for reason in [
            row['description'],
            row['calculations'],
            request_goal_answers.get('4'),
            request_goal_answers.get('5'),
            row['comment'],
            '\n'.join(self.get_chart_links(row)),
        ]:
            if reason:
                yield reason

    @staticmethod
    def get_chart_links(row):
        if row['chartLinks'] is None:
            return
        for link in row['chartLinks']:
            if link:
                yield link

    @staticmethod
    def get_request_goal_answers(row):
        if row['requestGoalAnswers'] and hasattr(row['requestGoalAnswers'], 'get'):
            return row['requestGoalAnswers']
        else:
            return {}

    def __call__(self, key, rows):
        last_row = None

        for row in rows:
            last_row = row

        if last_row is not None:
            row = {key: last_row.get(key) for key in self.valid_keys}
            if row["goalId"] is None and row["requestType"] == "GROWTH":
                row["goalId"] = 0
            elif row["goalId"] is None:
                row["goalId"] = -1
            row["version"] = "main"
            row["abc"] = last_row["abc"].lower()
            row["topAbc"] = self.get_enrich_abc(last_row["abc"]).get('topAbc')
            row["abcName"] = self.get_enrich_abc(last_row["abc"]).get('abcName')
            row["topAbcName"] = self.get_enrich_abc(last_row["abc"]).get('topAbcName')
            row["ticket"] = last_row["abcTicket"]
            row["name"] = last_row["summary"]
            row["preorderId"] = self.parse_preorder_id(last_row["preorderId"])
            row['resps'] = [last_row['responsible']]
            request_goal_answers = self.get_request_goal_answers(last_row)
            row["reason"] = '\n\n'.join(self.get_reasons(last_row, request_goal_answers))
            row["whatifno"] = request_goal_answers.get('6')
            if not hasattr(row['resources'], 'items'):
                return
            for cloud, resources in row['resources'].items():
                resources = {self.get_deadline(k): v for k, v in resources.items()}
                cloud_row = row.copy()
                cloud_row['cloud'] = cloud
                cloud_row['when'] = list(resources.keys())
                cloud_row['resources'] = resources
                yield cloud_row
            row["cloud"] = "ANY"
            row["clouds"] = ", ".join(row['resources'].keys())
            row["resources"] = "{}"
            yield row


class ReportReducer(object):
    def __call__(self, key, rows):
        cloud_rows = {}
        mcrp_cloud_ticket = {}
        for row in rows:
            if row['mcrpTicket'] is None:
                cloud_rows[row['cloud']] = row
            else:
                mcrp_cloud_ticket[row['cloud']] = row['mcrpTicket']
                if row['cloud'] not in cloud_rows:
                    cloud_rows[row['cloud']] = row
        for cloud, row in cloud_rows.items():
            row['mcrpTicket'] = mcrp_cloud_ticket.get(row['cloud'])
            row['resources'] = json.dumps(row['resources'])
            row['when'] = json.dumps(row['when'])
            yield row


class MarketMcrpReport(binary_task.LastBinaryTaskRelease, sdk2.Task):
    """MCRP Report task"""

    class Parameters(sdk2.Task.Parameters):
        # common parameters
        kill_timeout = 3600

        # custom parameters
        yt_cluster = sdk2.parameters.String("YT cluster", default="hahn", required=True)
        request_db_path = sdk2.parameters.String("MCRP Request YT db path", required=True)
        report_db_path = sdk2.parameters.String("MCRP Report YT db path", required=True)
        abc_request_db_path = sdk2.parameters.String("MCRP ABC Requests YT db directory path", required=True)
        abc_map_db_path = sdk2.parameters.String("MCRP ABC Map YT db directory path", required=True)
        tokens = sdk2.parameters.YavSecret("yav with yt token as 'yt' key and startrack token as 'st'", required=True)
        money_db_path = sdk2.parameters.String("MCRP Money db path", required=True)

        ext_params = binary_task.binary_release_parameters(stable=True)

    @staticmethod
    def get_db_path(db_path):
        return str(db_path).rstrip("/")

    @staticmethod
    def get_goals_mapping(yt_client, st_client, request_db):
        goals_base_url_host = 'goals.yandex-team.ru'

        def _get_goal_id(url):
            parsed = urlparse(url)
            if parsed.netloc == goals_base_url_host and 'goal=' in parsed.query:
                for i in parsed.query.split('&'):
                    if 'goal=' in i:
                        goalId = i.split('=')[1]
                        if goalId.isdigit():
                            return int(goalId)
                        else:
                            return None
            else:
                return None

        def _process_goal(goal):
            if goal:
                # Goals from goals.yandex-team.ru
                if goals_base_url_host in goal:
                    goalId = _get_goal_id(goal)
                    if goalId:
                        goals_mapping[goal]['goalId'] = goalId
                    else:
                        goals_mapping[goal]['goalId'] = -1
                        goals_mapping[goal]['goalName'] = "Нет цели"
                # Custom goals string(may be startrack or something...)
                else:
                    goals_mapping[goal]['goalId'] = -2
                    goals_mapping[goal]['goalName'] = goal

        def _get_st_keys():
            result_keys = [
                '{}-{}'.format('GOALZ', x.get('goalId'))
                for x in goals_mapping.values()
                if x.get('goalId') > 0
            ]
            return result_keys

        # Main process
        from collections import defaultdict
        goals_mapping = defaultdict(dict)
        logging.debug("checking {}".format(request_db.join('requests')))
        if yt_client.exists(request_db.join('requests')):
            logging.debug("directory {} exists".format(request_db.join('requests')))
            import yt.wrapper as yt
            import json
            from urlparse import urlparse

            # Getting all goals from requests table
            logging.debug("getting goals from yt")
            for row in yt_client.read_table(
                yt.TablePath(request_db.join('requests'), columns=['sourceMeta'])
            ):
                if row:
                    _process_goal(json.loads(row.get('sourceMeta', '{}')).get('goal'))
            logging.debug("goals_mapping: {}".format(goals_mapping))

            # Getting goals tickets
            keys = _get_st_keys()
            logging.debug("keys: {}".format(keys))

            # Getting issues from st GOALZ queue
            if len(keys) > 0:
                tickets = st_client.issues.find(keys=keys)
            else:
                return goals_mapping

            # Mapping id and name
            tickets_mapping = dict()
            for ticket in tickets:
                tickets_mapping[int(ticket.key.split('-')[1])] = ticket.summary
            logging.debug("tickets_mapping: {}".format(tickets_mapping))

            # Setting name into main dict
            for k in goals_mapping.keys():
                if goals_mapping[k].get('goalId') and goals_mapping[k].get('goalId') > 0:
                    goals_mapping[k]['goalName'] = tickets_mapping.get(goals_mapping[k]['goalId'])
            logging.debug("goals_mapping: {}".format(goals_mapping))
        return goals_mapping

    @staticmethod
    def get_abc_mapping(yt_client, abc_map_db):
        import yt.wrapper as yt

        abc_mapping = {}
        for row in yt_client.read_table(
            yt.TablePath(abc_map_db.join('top-services-map'))
        ):
            # TODO: change format of top-services-map table to strict rows without lists
            # Actualy it could be error in mapping of abc_service and abcName
            for i, abc_service in enumerate(row['children']):
                abc_mapping[abc_service] = {
                    'topAbc': row.get('abc'),
                    'topAbcName': row.get('abc_name'),
                    'abcName': row.get('children_names')[i]
                }
            else:
                abc_mapping[row.get('abc')] = {
                    'topAbc': row.get('abc'),
                    'topAbcName': row.get('abc_name'),
                    'abc': row.get('abc'),
                    'abcName': row.get('abc_name')
                }

        return abc_mapping

    def run_tickets_report(self, yt_client, st_client, request_db, report_db, abc_request_db, abc_map_db, money_db):
        import yt.wrapper as yt

        last_processed = yt_client.get(request_db.join("lastProcessed"))
        abc_request_sequence = yt_client.get(abc_request_db.join("sequence"))
        last_money_db_commit = yt_client.get(money_db.join("costs").join('@last_commit_timestamp'))
        money_db_commit = 0
        last_mcrp_processed = 0
        last_abc_processed = 0

        if yt_client.exists(money_db.join("lastCommit")):
            money_db_commit = yt_client.get(money_db.join("lastCommit"))
        if yt_client.exists(report_db.join("mcrpRequestsLastProcessed")):
            last_mcrp_processed = yt_client.get(report_db.join("mcrpRequestsLastProcessed"))
        if yt_client.exists(report_db.join("abcRequestLastProcessed")):
            last_abc_processed = yt_client.get(report_db.join("abcRequestLastProcessed"))

        if last_money_db_commit > money_db_commit:
            yt_client.run_sort(yt.TablePath(money_db.join("costs")),
                               destination_table=yt.TablePath(money_db.join("costs-static"), schema=COSTS_SCHEMA),
                               sort_by=['abcTicket'])

            yt_client.set(money_db.join("lastCommit"), last_money_db_commit)

        if last_processed > last_mcrp_processed or abc_request_sequence > last_abc_processed:
            abc_mapping = self.get_abc_mapping(yt_client, abc_map_db)
            goals_mapping = self.get_goals_mapping(yt_client, st_client, request_db)
            valid_keys = [v['name'] for v in REPORT_SCHEMA]

            with yt_client.TempTable(report_db) as mcrp_table, \
                    yt_client.TempTable(report_db) as abc_table:

                yt_client.run_map_reduce(
                    None,
                    McrpReducer(abc_mapping, goals_mapping, valid_keys),
                    yt.TablePath(request_db.join("requests")),
                    mcrp_table,
                    reduce_by="ticket",
                    sort_by=["ticket", "id"],
                )

                yt_client.run_map_reduce(
                    None,
                    AbcReducer(abc_mapping, goals_mapping, valid_keys),
                    yt.TablePath(abc_request_db.join("requests")),
                    abc_table,
                    reduce_by="abcID",
                    sort_by=["abcID", "id"],
                )

                yt_client.run_map_reduce(
                    None,
                    ReportReducer(),
                    [mcrp_table, abc_table],
                    yt.TablePath(report_db.join("requests"), schema=REPORT_SCHEMA),
                    reduce_by="ticket",
                )
            yt_client.set(report_db.join("mcrpRequestsLastProcessed"), last_processed)
            yt_client.set(report_db.join("abcRequestLastProcessed"), abc_request_sequence)
        else:
            logging.info("No new changes")

        yt_client.write_table(yt.TablePath(report_db.join("updated"), schema=UPDATED_SCHEMA), [
            {'type': 'abc', 'when': yt_client.get(abc_request_db.join("synced"))},
            {'type': 'abc_broken', 'when': yt_client.get(abc_request_db.join("broken"))},
            {'type': 'report', 'when': int(datetime.utcnow().strftime('%s'))},
        ], raw=False)

    def on_execute(self):
        import yt.wrapper as yt
        from startrek_client import Startrek

        st_base_url = "https://st-api.yandex-team.ru/"
        st_user_agent = 'marketMcrp'

        yt_client = yt.YtClient(proxy=self.Parameters.yt_cluster, token=self.Parameters.tokens.data()["yt"])
        yt_client.config['spec_overrides'] = {'acl': YT_ACL}
        st_client = Startrek(useragent=st_user_agent,
                             base_url=st_base_url,
                             token=self.Parameters.tokens.data()["st"])

        self.run_tickets_report(yt_client,
                                st_client,
                                yt.YPath(self.get_db_path(self.Parameters.request_db_path)),
                                yt.YPath(self.get_db_path(self.Parameters.report_db_path)),
                                yt.YPath(self.get_db_path(self.Parameters.abc_request_db_path)),
                                yt.YPath(self.get_db_path(self.Parameters.abc_map_db_path)),
                                yt.YPath(self.get_db_path(self.Parameters.money_db_path))
                                )
