# coding=utf-8

import logging

from sandbox import sdk2
from sandbox.sandboxsdk import environments
from sandbox.common.utils import get_task_link
from sandbox.projects.market.mobile.helpers import svn
from sandbox.projects.market.mobile.helpers import firebase


class MarketMobileDownloadRemoteConfigTask(sdk2.Task):
    """
    Download template from Firebase Remote Config server to arcadia repo.
    """
    class Requirements(sdk2.Task.Requirements):
        environments = [
            environments.PipEnvironment('rsa', '4.5', custom_parameters=['--upgrade-strategy only-if-needed']),
            environments.PipEnvironment('oauth2client', '4.1.3', custom_parameters=['--upgrade-strategy only-if-needed'])
        ]

    class Parameters(sdk2.Task.Parameters):
        # custom parameters
        project_id = sdk2.parameters.String('Project id', required=True)
        key = sdk2.parameters.YavSecret('Google Key', required=True)
        target_path = sdk2.parameters.String('Target directory with remote config file', required=True)
        author = sdk2.parameters.String('Author of commit', default='zomb-sandbox-rw', required=True)
        skip_check = sdk2.parameters.Bool('Skip precommit check. DO NOT SET \"TRUE\"', default=False)

    def on_execute(self):
        logging.info("Staring %s task with params: %s", self.type, self.Parameters)

        remote_config, e_tag = firebase.get_remote_config(self.Parameters.project_id, self.Parameters.key)
        self._commit_data_to_arcadia_if_changed(remote_config, e_tag)

    def _commit_data_to_arcadia_if_changed(self, config, e_tag):
        """Commit actual Firebase Remote Config template to arcadia repo.

        :param: Firebase Remote Config template to be committed.
        :param: ETag header to be committed.
        """
        commit_required = False
        svn_dir = svn.get_arcadia_svn_dir(self.Parameters.target_path)

        config_file_name = 'remoteConfig.json'
        saved_config = svn.read_file(svn_dir, config_file_name)
        if config != saved_config:
            svn.write_file(svn_dir, config_file_name, config)
            commit_required = True

        e_tag_file_name = 'eTag.txt'
        saved_e_tag = svn.read_file(svn_dir, e_tag_file_name)
        if e_tag != saved_e_tag:
            svn.write_file(svn_dir, e_tag_file_name, e_tag)
            commit_required = True

        if not commit_required:
            logging.info("Remote config and ETag has not changed since last run, nothing to commit. Exiting...")
            return

        message = """
        {} update Firebase Remote Config template and ETag\n{}
        """.format(
            self.type, get_task_link(self.id)
        )
        if self.Parameters.skip_check:
            message += "\nSKIP_CHECK"
        else:
            message += "\nSKIP_JSON_CHECK"
        svn.commit_to_arc(svn_dir, message, self.Parameters.author)
