# coding: utf-8

import logging
import requests
import datetime
import re

from sandbox import sdk2

ADMIN_BETA_URL = "https://admin.beta.m.soft.yandex-team.ru/api/v1/bluemarket/"
ADMIN_BETA_BRANCH_URL = "https://admin.beta.m.soft.yandex-team.ru/api/v1/bluemarket/{0}/"


class MarketMobileDeleteOldBetaBranches(sdk2.Task):

    def on_execute(self):
        logging.info("Starting %s task with params: %s",
                     self.type, self.Parameters)
        self.days_old_diff = 30
        self.today = datetime.datetime.today()
        self.__remove_old_branches()

    def __remove_old_branches(self):
        all_branches_info = self.__get_all_branches_info()
        all_branches = all_branches_info["branches"]

        logging.info(
            "All branches have been received. Quantity: %s", len(all_branches))

        for branch in all_branches:
            # Пропуск ветки, если имя ветки невалидно или ветка релизная
            if not branch.strip() or self.__is_release_branch(branch):
                continue

            try:
                branch_info = self.__get_branch_info(branch)
                if (self.__is_old_branch(branch_info)):
                    delete_response = self.__delete_branch(branch)
                    logging.info(
                        "The branch has been deleted: %s \nBeta response: %s", branch, delete_response)
            except Exception as e:
                logging.error("Unhandled exception: %s", e)

    def __is_release_branch(self, branch_name):
        # Проверяем на мэтч с релизной веткой iOS (регулярка для 1.1.1i)
        if re.match(r'^(\d+\.)(\d+.)(\d+i)', branch_name):
            return True

        # Проверяем на мэтч с релизной веткой Android (регулярка для 1.1a)
        if re.match(r'^(\d+\.)(\d+a)', branch_name):
            return True

        return False

    def __is_old_branch(self, branch_info):
        builds = branch_info["builds"]
        upload_time = self.today

        if len(builds) > 0:
            upload_time = datetime.datetime.strptime(
                builds[0].get("upload_time")[0:10], "%Y-%m-%d")

        # Разница между текущей датой и последним обновлением ветки >= days_old_diff
        if ((self.today - upload_time).days >= self.days_old_diff):
            return True

        return False

    def __delete_branch(self, branch_name):
        branch_url = self.__build_branch_url(branch_name)
        return requests.delete(branch_url, verify=False).json()

    def __get_all_branches_info(self):
        return requests.get(ADMIN_BETA_URL, verify=False).json()

    def __get_branch_info(self, branch_name):
        branch_url = self.__build_branch_url(branch_name)
        return requests.get(branch_url, verify=False).json()

    def __build_branch_url(self, branch_name):
        return ADMIN_BETA_BRANCH_URL.format(branch_name)
