# coding=utf-8

import logging
import os
import json

from sandbox import sdk2
from sandbox.common.types import notification as ctn
from sandbox.common.utils import get_task_link
from sandbox.projects.common.juggler import jclient
from sandbox.sandboxsdk import environments
from sandbox.projects.common.arcadia import sdk as arcadia_sdk
from sandbox.projects.market.mobile.helpers import svn
from sandbox.projects.market.mobile.helpers import firebase
from difflib import unified_diff


class MarketMobileUploadRemoteConfigTask(sdk2.Task):
    """
    Upload template from arcadia repo to Firebase Remote Config server.
    """
    class Requirements(sdk2.Task.Requirements):
        environments = [
            environments.PipEnvironment('rsa', '4.5', custom_parameters=['--upgrade-strategy only-if-needed']),
            environments.PipEnvironment('oauth2client', '4.1.3', custom_parameters=['--upgrade-strategy only-if-needed'])
        ]

    class Parameters(sdk2.Task.Parameters):
        # custom parameters
        checkout_arcadia_from_url = sdk2.parameters.ArcadiaUrl(
            'Arcadia URL',
            default='arcadia-arc:/#trunk',
            required=True
        )
        project_id = sdk2.parameters.String('Project id', required=True)
        key = sdk2.parameters.YavSecret('Google Key', required=True)
        source_path = sdk2.parameters.String('Target directory with remote config file', required=True)
        notification_recipients = sdk2.parameters.List('Recipient emails', default=[], required=False)
        juggler_host = sdk2.parameters.String('Juggler host', required=True)
        validate_only = sdk2.parameters.Bool('Only validate template without publishing', default=False)

    def on_execute(self):
        logging.info("Starting %s task with params: %s", self.type, self.Parameters)

        with arcadia_sdk.mount_arc_path(self.Parameters.checkout_arcadia_from_url) as arcadia_src_dir:
            logging.info("Arcadia src dir {}".format(arcadia_src_dir))
            config_src_dir = os.path.join(arcadia_src_dir, self.Parameters.source_path)
            logging.info("Config src dir {}".format(config_src_dir))
            arcadia_config = svn.read_file(config_src_dir, 'remoteConfig.json')
            e_tag = svn.read_file(config_src_dir, 'eTag.txt')

        if self.Parameters.validate_only:
            firebase.post_remote_config(
                self.Parameters.project_id, self.Parameters.key, arcadia_config, e_tag, self.Parameters.validate_only
            )
        else:
            old_config, _ = firebase.get_remote_config(self.Parameters.project_id, self.Parameters.key)
            _ = firebase.post_remote_config(
                self.Parameters.project_id, self.Parameters.key, arcadia_config, e_tag, self.Parameters.validate_only
            )
            subject, body = self._make_notification_content(old_config, arcadia_config)
            self._send_email_notification(subject, body)
            self._send_juggler_notification(subject, body)

    def _send_email_notification(self, subject, body):
        """Send email notification with configs diff.

        :param: Old config before upload.
        :param: New config after upload.
        """
        logging.info("Sending diff report to emails: {}".format(self.Parameters.notification_recipients))
        self.server.notification(
            subject=subject,
            body=body,
            recipients=self.Parameters.notification_recipients,
            transport=ctn.Transport.EMAIL,
            charset=ctn.Charset.UTF
        )

    def _send_juggler_notification(self, subject, body):
        """Send Juggler notification with configs diff.

        :param: Old config before upload.
        :param: New config after upload.
        """
        logging.info("Sending diff report to Juggler host: {}".format(self.Parameters.juggler_host))
        message = '\n\n'.join([subject, body])
        jclient.send_events_to_juggler(self.Parameters.juggler_host, 'remote_config_release', 'WARN', message)

    def _make_notification_content(self, old_config, new_config):
        new_config_json = json.loads(new_config)

        subject = [
            "#firebase_remote_config_release",
            "RemoteConfig v.{} успешно загружен в Firebase.".format(new_config_json["version"]["versionNumber"]),
            "Sandbox Task: {}".format(get_task_link(self.id))
        ]

        delimeter = '____________________________________________________\n'
        diff = unified_diff(old_config.splitlines(), new_config.splitlines(), n=1)
        body = []
        for d in diff:
            if d.startswith('---') or d.startswith('+++') or len(d) == 0:
                continue
            if d.startswith('@@'):
                body.append(delimeter)
            else:
                body.append(d)
        body.append(delimeter)
        body.insert(0, '\nИзменения:')

        return '\n'.join(subject), '\n'.join(body)
