import logging
import requests

from sandbox.sandboxsdk.errors import SandboxTaskFailureError


BASE_URL = 'https://firebaseremoteconfig.googleapis.com'
SCOPES = ['https://www.googleapis.com/auth/firebase.remoteconfig']


def get_remote_config(project_id, service_key):
    """Retrieve an actual Remote Config template from Firebase Server.

    :param: Firebase project id.
    :param: Firebase service account key from Yav.
    :return: Firebase Remote Config template.
    """
    url = _build_remote_config_url(project_id)
    headers = _build_remote_config_headers(service_key.data())
    resp = requests.get(url, headers=headers)
    if resp.status_code == 200:
        logging.info("Template has been downloaded. Response text {}".format(resp.text))
        remote_config = resp.text
        e_tag = resp.headers['ETag']
    else:
        raise SandboxTaskFailureError(
            "Unable to download template. Response text {}".format(resp.text)
        )
    return remote_config, e_tag


def post_remote_config(project_id, service_key, config, e_tag, validate_only):
    """Post Firebase Remote Config template.

    :param: Firebase project id.
    :param: Firebase service account key from Yav.
    :param: Firebase Remote Config template to be uploaded.
    :param: ETag header from last update.
    :param: Only validate without publishing.
    :return: Firebase Remote Config template.
    """
    url = _build_remote_config_url(project_id)
    headers = _build_remote_config_headers(service_key.data())
    headers.update({'If-Match': e_tag})
    params = {'validateOnly': validate_only}
    resp = requests.put(url, headers=headers, params=params, data=config)
    if resp.status_code == 200:
        logging.info("Template has been uploaded. (validateOnly = {}) Response text:\n{}\nETag: {}".format(
            validate_only, resp.text, resp.headers["ETag"]
        ))
        remote_config = resp.text
    else:
        raise SandboxTaskFailureError(
            "Unable to publish template. Response text {}".format(resp.text)
        )
    return remote_config


def _build_remote_config_url(project_id):
    remote_config_endpoint = 'v1/projects/' + project_id + '/remoteConfig'
    return BASE_URL + '/' + remote_config_endpoint


def _build_remote_config_headers(service_key):
    return {
        'Authorization': 'Bearer ' + _get_access_token(service_key),
        'Accept-Encoding': 'gzip',
        'Content-Type': 'application/json; UTF-8'
    }


def _get_access_token(service_key):
    """Retrieve a valid access token that can be used to authorize requests.

    :param: Service key dict.
    :return: Access token.
    """
    from oauth2client.service_account import ServiceAccountCredentials
    if not service_key:
        raise SandboxTaskFailureError(
            "Service key does not exists"
        )
    credentials = ServiceAccountCredentials.from_json_keyfile_dict(service_key, SCOPES)
    access_token_info = credentials.get_access_token()
    return access_token_info.access_token
