import os
import logging

from sandbox import sdk2
from sandbox.sandboxsdk.errors import SandboxTaskFailureError


def get_arcadia_svn_dir(path):
    """Retrieve an arcadia svn directory where are stored template and etag files.

    :param: url of Arcadia to be checkouted
    :param: path to target dir in Arcadia
    :return: Svn dir string.
    """
    svn_url = sdk2.svn.Arcadia.trunk_url(path)
    svn_dir = sdk2.svn.Arcadia.get_arcadia_src_dir(svn_url)
    if not svn_dir:
        raise SandboxTaskFailureError(
            "Path \"{}\" set by parameter is empty or does not exists".format(
                path
            )
        )
    return str(svn_dir)


def read_file(svn_dir, file_name):
    """Retrieve file content saved in arcadia repo.

    :svn_dir: Arcadia svn dir string.
    :file_name: Name of file to be read.
    :return: File content string
    """
    full_path = os.path.join(svn_dir, file_name)
    if not os.path.isfile(full_path):
        logging.info("File does not exist in arcadia: {}".format(full_path))
        return None
    with open(full_path, 'r') as f:
        content = f.read()
    if not content:
        raise SandboxTaskFailureError(
            "Read content of file \"{}\" from arcadia dir \"{}\" failed or file is empty.".format(file_name, svn_dir)
        )
    return content


def write_file(svn_dir, file_name, content):
    """Update file content saved in arcadia repo.

    :svn_dir: Arcadia svn dir string.
    :file_name: Name of file to be read.
    :content: Content string to write.
    """
    full_path = os.path.join(svn_dir, file_name)
    with open(full_path, 'w') as f:
        f.write(content)


def commit_to_arc(svn_dir, message, author):
    """Commit changes to arcadia repo.

    :svn_dir: Arcadia svn dir string.
    :comment: Commit message.
    :author: Author of commit.
    """
    try:
        sdk2.svn.Arcadia.commit(svn_dir, message, author)
    except Exception as e:
        logging.info("Code will be merged when required checks would be passed")
        logging.info(e)
