# -*- coding: utf-8 -*-

from sandbox.projects.market.mqm.ImportErrorsFromStartrek.common import PlanFactRepositoryClient, \
    StartrekClient
from sandbox import sdk2
import sandbox.projects.common.environments as env
from datetime import datetime, timedelta
import logging

BUSINESS_ERROR = "BUSINESS_ERROR"


REPORT_TEMPLATE = """<table>
   <tr>
     <td> Dry run </td>
     <td> {{dry_run}} </td>
   </tr>
   <tr>
     <td> Tickets interval </td>
     <td> {{interval}} </td>
   </tr>
   <tr>
     <td> Found issues </td>
     <td> {{issues_count}} </td>
   </tr>
   <tr>
     <td> Updated plan-facts </td>
     <td> {{total_updated_plan_facts}} </td>
   </tr>
</table>"""


QUEUE_TO_PLAN_FACT_PRODUCER = {
    'MQMDELIVERYDSSC': ["DropshipScIntakePlanFactProcessor"],
    'MQMDELIVERYFFSD': ["FfDsIntakePlanFactProcessor"],
    'MQMDELIVERYMK': ["ScDsIntakePlanFactProcessor", "RecipientCourierPlanFactProcessor", "OnDemandShipmentPlanFactProcessor", "FfDsIntakePlanFactProcessor"],
    'MQMDELIVERYDSSD': ["DropshipDsIntakePlanFactProcessor"],
    'MQMSHIPMENTSC': ["ScShippedPlanFactProcessor"],
    'MQMDELIVERYSCSD': ["ScDsIntakePlanFactProcessor"],
    'MQMASSEMBLYFF': ["FfPreparedPlanFactProcessor"],
    'MQMSHIPMENTFF': ["FfShippedPlanFactProcessor"],
    'MQMFINALSTATUS': ["DeliveredPlanFactProcessor", "OnDemandFinalStatusPlanFactProcessor"],
}


class ImportErrorsFromStartrek(sdk2.Task):
    """Класс ImportErrorsFromStartrek используется для импорта причин из полей тикетов в бд.

    Логика работы:
        1. Из указанной очереди в рамках указанного диапазона выгружаются созданные тикеты.
        2. Выбираются тикеты, в которых есть есть причина.
        3. Для каждого тикета в бд находится соответствующая группа, в которую слинкованы план-факты.
        4. Для всех план-факты найденной группы проставляется причина и подпричина.

    В режиме "dry run" вместо обновления на шаге 4 будет добавлен лог.

    Интервал выгрузки тикетов из очереди можно задать двумя способами:
        1. Явно указать интервал [time_range_start, time_range_end].
        2. Указать количество последних дней scan_last_days,
            что преобразуется в [today - scan_last_days, today].
        Способ 1 основной. Если параметры time_range_start и time_range_end пусты,
        то используется 2й способ.

    Для работы скрипта также нужно указать поле, в котором находится причина.
    Это можно узнать из json, выгрузив данные через api,
    например https://st-api.yandex-team.ru/v2/issues/MQMSHIPMENTDS-10549.
    Ключ "602c3d42f7d3b93bbc6c6704--problemReason" используется для всех тикетов этой очереди.
    У каждой очереди свой ключ.
    """

    class Requirements(sdk2.Requirements):
        environments = (
            env.PipEnvironment('psycopg2-binary', '2.7.7'),
            env.PipEnvironment('startrek_client', custom_parameters=['requests==2.18.4']),
            env.PipEnvironment('Jinja2', '2.8', use_wheel=True),
        )

    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 300
        dry_run = sdk2.parameters.Bool(
            "Dry run",
            default=True,
            required=True,
        )

        secret = sdk2.parameters.YavSecret(
            "Job secrets",
            default="sec-01f53zy43pqekgpae85b3sb0h2",
            required=True,
        )

        queue_name = sdk2.parameters.String(
            "Queue name",
            default="",
            required=True
        )

        time_range_start = sdk2.parameters.StrictString(
            "Start date for scan, YYYY-MM-DD",
            regexp="(^\\d{4}-\\d{2}-\\d{2}$|^$)",
        )

        time_range_end = sdk2.parameters.StrictString(
            "End date for scan, YYYY-MM-DD",
            regexp="(^\\d{4}-\\d{2}-\\d{2}$|^$)",
        )

        scan_last_days = sdk2.parameters.Integer(
            "Last days for scan",
            default=1,
        )

        """
            MQMDELIVERYDSSC -> 60533e0dc332be363a710949--ProblemReason
            MQMDELIVERYFFSD -> 608659f3644b9007fba04512--problemReason
            MQMDELIVERYMK -> 6062ef16de335c41c110ae34--ProblemReason
            MQMDELIVERYDSSD -> 60865ad1e33dcf43794f8a71--problemReason
            MQMSHIPMENTSC -> 60a3dbe8c250f41357c5f3ce--ProblemReason
            MQMDELIVERYSCSD -> 60865a64e33dcf43794f89ab--problemReason
            MQMASSEMBLYFF -> incidentReason
            MQMSHIPMENTFF -> 60d4a70f084e0630e5d06282--ProblemReason
            MQMFINALSTATUS -> incidentReason
        """
        ticket_reason_field = sdk2.parameters.String(
            "Field where reason is stored",
            required=True,
            default="",
        )

    def calculate_interval(self):
        if self.Parameters.time_range_start != "" and self.Parameters.time_range_end != "":
            return self.Parameters.time_range_start, self.Parameters.time_range_end
        elif self.Parameters.scan_last_days is not None and self.Parameters.scan_last_days > 0:
            end = datetime.today()
            start = end - timedelta(days=self.Parameters.scan_last_days)
            return start.strftime("%Y-%m-%d"), end.strftime("%Y-%m-%d")
        else:
            raise ValueError('Invalid interval arguments')

    def save_context(self, issues_count, total_updated_plan_facts):
        self.Context.artifact_suites_result = dict(
            issues_count=issues_count,
            total_updated_plan_facts=total_updated_plan_facts,
        )
        self.Context.save()

    @sdk2.report(title="Export result")
    def report(self):
        import jinja2
        return jinja2.Environment().from_string(REPORT_TEMPLATE).render(self.Context.artifact_suites_result)

    def on_execute(self):
        startrek_client = StartrekClient()
        startrek_client.connect(self.Parameters.secret.data()['startrek_token'])
        db_client = PlanFactRepositoryClient()
        db_client.connect(self.Parameters.secret.data()['database_connection_url'])

        interval_start, interval_end = self.calculate_interval()
        logging.info("Issues search interval {}".format([interval_start, interval_end]))
        issues = startrek_client.get_issues_with_reason(
            queue_name=self.Parameters.queue_name,
            date_start=interval_start,
            date_end=interval_end,
            reason_field=self.Parameters.ticket_reason_field,
        )
        logging.debug("Found issues {}".format(issues))
        logging.info("Found issues {}".format(len(issues)))
        total_updated_plan_facts = 0
        for issue in issues:

            logging.info("Found orders {} for issue {} with reason \"{}\"".format(issue.orders, issue.key, issue.reason))

            producers = QUEUE_TO_PLAN_FACT_PRODUCER[self.Parameters.queue_name]
            if not self.Parameters.dry_run:
                updated_plan_facts = db_client.update_plan_facts_subcause(
                    cause=BUSINESS_ERROR,
                    subcause=issue.reason,
                    orders=issue.orders,
                    producers=producers,
                )
                logging.debug("Plan-facts {} updated for order {} and producers {}.".format(updated_plan_facts, issue.orders, producers))
                total_updated_plan_facts += len(updated_plan_facts)
            else:
                read_plan_facts = db_client.select_plan_facts(
                    orders=issue.orders,
                    producers=producers,
                )
                logging.debug("Plan-facts {} read for order {} and producers {}.".format(read_plan_facts, issue.orders, producers))
                total_updated_plan_facts += len(read_plan_facts)

        self.Context.artifact_suites_result = dict(
            dry_run=self.Parameters.dry_run,
            interval=[interval_start, interval_end],
            issues_count=len(issues),
            total_updated_plan_facts=total_updated_plan_facts,
        )
        self.Context.save()
