# -*- coding: utf-8 -*-

from sandbox.projects.market.mqm.ImportErrorsFromStartrek.sql import SET_CAUSE_FOR_PLAN_FACTS_SQL
import logging
from collections import namedtuple


ParsedStartrekIssue = namedtuple('ParsedStartrekIssue', ['key', 'orders', 'reason'])


def _has_reason_field(issue, field_name):
    return field_name in issue.__dict__["_value"]


def _get_reason_field(issue, field_name):
    return (issue.__dict__["_value"][field_name]).lower().strip()


def get_orders_field(issue):
    return [str(order.strip()) for order in issue.customerOrderNumber.split(',')]


def prepare_sql(sql_template, sql_vars={}):
    import jinja2
    logging.debug("Resolving template {} with args {}.".format(sql_template, sql_vars))
    return jinja2.Environment().from_string(sql_template).render(sql_vars)


def _execute_sql(database_connection, sql_template, sql_vars):
    from psycopg2.extras import DictCursor
    from psycopg2._psycopg import DatabaseError
    try:
        sql_query = prepare_sql(sql_template, sql_vars)
        cur = database_connection.cursor(cursor_factory=DictCursor)

        logging.debug("Executing SQL: {}".format(sql_query))
        cur.execute(sql_query)

        return cur
    except DatabaseError as e:
        logging.error("Failed to make sql query", e)
        raise e


class StartrekClient(object):
    def __init__(self):
        self.startrek_connection = None

    def connect(self, token):
        from startrek_client import Startrek
        self.startrek_connection = Startrek(
            useragent="",
            base_url="https://st-api.yandex-team.ru/v2/",
            token=token
        )

    def get_issues_with_reason(self, queue_name, date_start, date_end, reason_field):
        issues = self.startrek_connection.issues.find(
            filter={
                'queue': queue_name,
                'created': {
                    'from': date_start,
                    'to': date_end
                }
            },
            order=['key'],
            per_page=1000,
        )
        logging.debug("Found issues {} from range".format(len(issues)))
        issues = list(filter(lambda issue: _has_reason_field(issue, reason_field) and issue.customerOrderNumber is not None, issues))
        logging.debug("Found issues with reason and orderes {}".format(len(issues)))
        parsed_issues = [
            ParsedStartrekIssue(
                key=issue.key,
                reason=_get_reason_field(issue, reason_field),
                orders=get_orders_field(issue),
            ) for issue in issues]
        return parsed_issues


class PlanFactRepositoryClient(object):
    def __init__(self):
        self.db_connection = None

    def connect(self, connection_url):
        import psycopg2
        self.db_connection = psycopg2.connect(connection_url, target_session_attrs='read-write')
        self.db_connection.autocommit = True

    def update_plan_facts_subcause(self, orders, producers, cause, subcause):
        cursor = _execute_sql(
            self.db_connection,
            SET_CAUSE_FOR_PLAN_FACTS_SQL,
            sql_vars={
                "orders": orders,
                "producers": producers,
                "cause": cause,
                "subcause": subcause,
            }
        )
        plan_facts_id = []
        for row in cursor:
            plan_facts_id.append(row["id"])
        return plan_facts_id

    def select_plan_facts(self, orders, producers):
        cursor = _execute_sql(
            self.db_connection,
            SET_CAUSE_FOR_PLAN_FACTS_SQL,
            sql_vars={
                "orders": orders,
                "producers": producers,
            }
        )
        plan_facts_id = []
        for row in cursor:
            plan_facts_id.append(row["id"])
        return plan_facts_id
