# coding=utf-8

import logging

from sandbox import sdk2
from sandbox.sandboxsdk import environments
from sandbox.projects.common.juggler import jclient
from datetime import datetime, timedelta, date
from itertools import groupby

days_from_cancellation = 32
common_missing_orders_title = "будет выставлена претензия за возврат заказов больше 32 дней"


class StartrekIssueYtComparatorMonitoring(sdk2.Task):
    # список параметров для задачи
    class Parameters(sdk2.Task.Parameters):
        yql_token_name = sdk2.parameters.String(
            "YQL token name",
            default='YQL_TOKEN',
            required=True)

        st_user_agent = sdk2.parameters.String('Startrack user agent', required=True)
        st_token_vault_key = sdk2.parameters.String('Startrack token vault key', required=True)
        st_url = sdk2.parameters.String('Startrack url', required=True)

        with sdk2.parameters.Group("Juggler") as juggler_group:
            send_juggler_events = sdk2.parameters.Bool(
                "Send juggler events",
                default_value=False
            )

            with send_juggler_events.value[True]:
                juggler_host_all = sdk2.parameters.String("Juggler host all partners")
                juggler_service_all = sdk2.parameters.String("Juggler service all partners")
                juggler_host_partner = sdk2.parameters.String("Juggler host by partner")
                juggler_service_partner = sdk2.parameters.String("Juggler service by partner")

    class Requirements(sdk2.Requirements):
        disk_space = 1024 * 4
        environments = (environments.PipEnvironment('yandex-yt'),
                        environments.PipEnvironment('yql'),
                        environments.PipEnvironment('startrek_client', custom_parameters=['requests==2.18.4']))

    # Получить номера заказов, которые остутствуют в тикете с общими номерами заказов отмененных больше N дней
    def get_mismatch_in_order_ids(self, yql_token, startrek_token):
        order_ids = self.get_order_ids_greater_than_Ndays(yql_token)
        issues = self.get_issue_with_not_returned_ids(startrek_token)
        if len(issues) == 0:
            logging.error("Unable to find issue with common missing order ids")
            return []

        declared_order_ids = self.extract_order_ids_from(issues[0])
        market_fault_order_ids = self.get_market_fault_order_ids(yql_token, declared_order_ids)
        not_returned_order_ids = set(order_ids) - set(market_fault_order_ids)

        # Идентификаторы из YT которых нет в issue startrek
        return set(not_returned_order_ids) - set(declared_order_ids)

    def get_order_ids_greater_than_Ndays(self, yql_token):
        from yt import wrapper
        from yt import clickhouse as chyt

        client = wrapper.YtClient(proxy='hahn', token=yql_token)
        yql_query = ("SELECT DISTINCT(order_id) "
                     "FROM `//home/market/production/analytics/business/delivery/claim/order_not_returned` "
                     "WHERE days_from_cancellation > " + str(days_from_cancellation) + " "
                     "AND (claim_status_compensation_sd NOT IN ('Compensation_from_SD', 'Compensation_from_SD_unpaid',"
                     " 'Compensation_from_SD_overpaid', 'Compensation_from_SD_waiting')"
                     " OR claim_status_compensation_sd IS NULL) "
                     "AND order_logistic_scheme_name IN ('ДШ-СЦ-СД', 'ДШ-СД', 'ДШ-ДО-СЦ-СД') "
                     "AND first_shipment_warehouse_datetime > '2022-03-01T00:00:00Z' "
                     "AND position_status IN ('В прямом транзите', 'В возвратном транзите');")

        mapped = list(chyt.execute(yql_query, alias='*ch_public', client=client))
        logging.info("Found order ids: " + str(len(mapped)))

        return list(map(lambda s: s['order_id'], mapped))

    def get_market_fault_order_ids(self, yql_token, order_ids):
        from yt import wrapper
        from yt import clickhouse as chyt

        client = wrapper.YtClient(proxy='hahn', token=yql_token)
        logging.info(order_ids)
        joining_order_ids = ", ".join(str(order_id) for order_id in order_ids)
        logging.info(joining_order_ids)
        yql_query = ("SELECT DISTINCT(`id`) "
                     "FROM `//home/market/production/oklo/prod/claim_sd/YT_JUDGE_DREDD_transactions_mqm` "
                     "WHERE status = 'MARKET_FAULT' and id in (" + joining_order_ids + ")")

        logging.info(yql_query)
        mapped = list(chyt.execute(yql_query, alias='*ch_public', client=client))
        logging.info("Found MARKET_FAULT order ids: " + str(len(mapped)))

        return list(map(lambda s: s['id'], mapped))

    def get_issue_with_not_returned_ids(self, startrek_token):
        from startrek_client import Startrek
        st = Startrek(useragent=self.Parameters.st_user_agent,
                      base_url=self.Parameters.st_url,
                      token=startrek_token)

        # даты понедельника
        dt = date.today()
        this_week_monday = dt - timedelta(days=dt.weekday())
        this_week_monday_formatted = str(this_week_monday)

        issues = st.issues.find('Queue: MQMCLAIM' +
                                ' Summary: "' + common_missing_orders_title + '"' +
                                ' Created: >' + this_week_monday_formatted + ' ' +
                                ' "Sort By": Key DESC',
                                per_page=1000)

        if len(issues) != 0:
            logging.info("Found issue with common missing order ids. Key="+issues[0]['key'])

        return issues

    def extract_order_ids_from(self, issue):
        import re

        result = re.search(r":([\w\s,]+)Общая сумма", str(issue['description']))

        order_ids_from_issue = result.group(1) if result is not None else ""
        if order_ids_from_issue == "":
            return []

        declared_order_ids = map(lambda a: int(a), order_ids_from_issue.split(","))
        logging.info("Declared order ids")
        logging.info(declared_order_ids)
        return declared_order_ids

    # Получение информации о пропущенных order id для каждого парнтера
    def get_errors_mismatch_partner(self, yql_token, startrek_token):
        declared_order_ids = self.get_partner_order_ids(yql_token)
        partner_issues = self.get_partner_issues(startrek_token)
        return self.get_mismatch_partner_order_ids(partner_issues, declared_order_ids)

    def get_partner_order_ids(self, yql_token):
        from yt import wrapper
        from yt import clickhouse as chyt

        client = wrapper.YtClient(proxy='hahn', token=yql_token)

        # даты понедельника и воскресенья предыдущей недели
        dt = date.today() - timedelta(days=7)
        start = dt - timedelta(days=dt.weekday())
        end = start + timedelta(days=6)

        start_datetime = datetime.combine(start, datetime.min.time())
        end_datetime = datetime.combine(end, datetime.max.time())

        yql_query = ("SELECT DISTINCT(order_id, partner_id) as order_id_partner_id"
                     " FROM `//home/market/production/analytics/business/delivery/claim/order_partners_for_claim`"
                     " WHERE date_claim > '" + start_datetime.isoformat() + "'"
                     " AND date_claim <= '" + end_datetime.isoformat() + "'"
                     " AND order_claim_status IS NULL"
                     " AND partner_claim_flag = '1'"
                     " AND partner_subtype_name in ('Партнерские ПВЗ (ИПэшники)')"
                     " AND position_status in ('В прямом транзите', 'В возвратном транзите')")

        mapped = list(chyt.execute(yql_query, alias='*ch_public', client=client))
        logging.info("Found partner order ids: " + str(len(mapped)))
        partner_order_id_pairs = list(map(lambda s: s['order_id_partner_id'], mapped))

        def get_partner_id(k):
            return k[1]

        partner_order_id_pairs = sorted(partner_order_id_pairs, key=get_partner_id)
        result = {}

        for key, value in groupby(partner_order_id_pairs, get_partner_id):
            result[key] = list(v[0] for v in value)

        return result

    def get_mismatch_partner_order_ids(self, issues, yt_partner_order_ids):
        errors = []
        for issue in issues:
            (partner_id, order_ids) = self.extract_partner_id_order_ids(issue)
            if partner_id is None or order_ids is None:
                continue

            if partner_id in yt_partner_order_ids:
                missing_order_ids = [x for x in yt_partner_order_ids[partner_id] if x not in order_ids]

                if len(missing_order_ids) != 0:
                    missing_order_ids_str = ", ".join(map(lambda x: str(x), missing_order_ids))
                    errors.append("Partner with id=" + str(partner_id) + " missing order_ids=" + missing_order_ids_str)

        if len(errors) == 0:
            return ''
        else:
            return '\n'.join(errors)

    def extract_partner_id_order_ids(self, issue):
        partner_id = None
        order_ids = None

        try:
            partner_id = self.__extract_partner_id_or_None__(issue['summary'])
            order_ids = self.__extract_order_ids_or_None__(issue['description'])
        except:
            logging.error("Unable to parse partner issue " + issue['key'])

        if partner_id is None or order_ids is None:
            logging.error("Unable to parse partner issue " + issue['key'])

        return partner_id, order_ids

    def __extract_partner_id_or_None__(self, issue_summary):
        import re

        summary_match = re.search(r"Партнеру ([0-9]*) будет", str(issue_summary))
        matched_partner_id = summary_match.group(1) if summary_match is not None else ""
        if matched_partner_id == "":
            return None

        return int(matched_partner_id)

    def __extract_order_ids_or_None__(self, issue_description):
        import re

        description_match = re.search(r"Заказы в выставленной претензии:([\w\s,]+)", str(issue_description))
        matched_order_ids = description_match.group(1) if description_match is not None else ""
        if matched_order_ids == "":
            return None

        return list(map(lambda a: int(a), matched_order_ids.split(",")))

    def get_partner_issues(self, startrek_token):
        from startrek_client import Startrek
        st = Startrek(useragent=self.Parameters.st_user_agent,
                      base_url=self.Parameters.st_url,
                      token=startrek_token)

        dt = date.today()
        monday = (dt - timedelta(days=dt.weekday()))
        monday_formatted = monday.strftime("%Y-%m-%d")
        sunday = (monday + timedelta(days=6)).strftime("%Y-%m-%d")

        issues = st.issues.find('Queue: MQMCLAIM' +
                                ' Summary: Партнеру' +
                                ' Status: !cancelled ' +
                                ' Created: "' + monday_formatted + '".."' + sunday + '" ' +
                                ' "Sort By": Key DESC',
                                per_page=1000)

        return issues

    def common_push_event(self, status, message):
        juggler_host = self.Parameters.juggler_host_all
        juggler_service = self.Parameters.juggler_service_all
        self.send_juggler_event(
            status=status,
            description=message.format(sdk2.Task.current.id),
            juggler_host=juggler_host,
            juggler_service=juggler_service)

    def partner_push_event(self, status, message):
        juggler_host = self.Parameters.juggler_host_partner
        juggler_service = self.Parameters.juggler_service_partner
        self.send_juggler_event(
            status=status,
            description=message.format(sdk2.Task.current.id),
            juggler_host=juggler_host,
            juggler_service=juggler_service)

    def send_juggler_event(self, status, description, juggler_host, juggler_service):
        if not self.Parameters.send_juggler_events:
            return

        if not juggler_host or not juggler_service:
            return

        jclient.send_events_to_juggler(juggler_host, juggler_service, status, description)
        self.Context.juggler_status = status
        self.set_info("Send juggler event. Host='{}', service='{}', status='{}', description='{}'".format(
            juggler_host,
            juggler_service,
            status,
            description
        ))

    def on_execute(self):
        yql_token = sdk2.Vault.data(self.Parameters.yql_token_name)
        startrek_token = sdk2.Vault.data(self.Parameters.st_token_vault_key)
        common_missing_orders = self.get_mismatch_in_order_ids(yql_token, startrek_token)
        if len(common_missing_orders) == 0:
            logging.info("OK")
            self.common_push_event("OK", "")
        else:
            common_missing_orders_message = ",".join(str(id) for id in common_missing_orders)
            logging.info("CRIT " + common_missing_orders_message)
            self.common_push_event("CRIT", common_missing_orders_message)

        partner_missing_orders = self.get_errors_mismatch_partner(yql_token, startrek_token)
        logging.info("partner_missing_orders " + partner_missing_orders)
        if partner_missing_orders == '':
            self.partner_push_event("OK", "")
        else:
            logging.info("CRIT")
            self.partner_push_event("CRIT", partner_missing_orders)
