# -*- coding: utf-8 -*-

import os
from sandbox import sdk2
import tarfile
from sandbox.sdk2.vcs.git import Git
from sandbox.sandboxsdk import process
from sandbox.projects import resource_types
from sandbox import common
import sandbox.common.types.client as ctc
import sandbox.common.types.task as ctt
from sandbox.projects.BuildDockerImageV6 import BuildDockerImageV6


class MarketMultitestingBuildDockerImage(sdk2.Task):
    """
        Собирает Docker-образ используя Ansible.
        https://st.yandex-team.ru/MARKETINFRA-1575

        Это страшный легаси, который умрёт с переездом мультитестингов в RTC.
        Ansible здесь используется только для шаблонизации файлов, из которых собираются образы.
    """

    class Requirements(sdk2.Task.Requirements):
        client_tags = ctc.Tag.GENERIC
        disk_space = 10 * 1024  # MB

    class Parameters(sdk2.Task.Parameters):
        git_repository = sdk2.parameters.Url(
            'Репозиторий',
            default='https://github.yandex-team.ru/market-at/infrastructure.git',
            required=True
        )
        branch = sdk2.parameters.String('Ветка репозитория', default='master', required=True)
        yml_file_path = sdk2.parameters.String('YML-файл, который нужно запустить', required=True)
        image_name = sdk2.parameters.String('Название Docker-образа', required=True)
        image_tag = sdk2.parameters.String('Тег Docker-образа', required=True)
        host_group = sdk2.parameters.String('Название группы хостов для ансибла')
        parameters = sdk2.parameters.Dict('Параметры плейбука')

    class Context(sdk2.Task.Context):
        children = []

    def on_execute(self):
        with self.memoize_stage.stage1(commit_on_entrance=False):
            repository_directory = self.clone_repository()
            ansible_directory = self.prepare_ansible()
            image_sources_directory = self.generate_image_sources(
                ansible_directory=ansible_directory,
                repository_directory=repository_directory
            )
            image_sources_resource_id = self.create_image_sources_resource(
                image_sources_directory=image_sources_directory
            )
            self.build_docker_image(image_sources_resource_id)

        with self.memoize_stage.stage2(commit_on_entrance=False):
            if not all(task.status in ctt.Status.Group.SUCCEED for task in self.find(id=self.Context.children)):
                raise common.errors.TaskFailure("Not all Docker containers built (see subtasks for details)")

    def clone_repository(self):
        directory = self.path('repository').as_posix()
        Git(self.Parameters.git_repository).clone(directory, self.Parameters.branch)
        return directory

    def prepare_ansible(self):
        resource_path = sdk2.ResourceData(sdk2.Resource.find(id='325191247').first()).path.as_posix()
        ansible_directory = self.path('ansible').as_posix()
        tarfile.open(resource_path).extractall(ansible_directory)
        return ansible_directory

    def generate_image_sources(self, ansible_directory, repository_directory):
        image_sources_directory = self.path('image_sources').as_posix()
        os.mkdir(image_sources_directory)

        hosts_file_path = os.path.join(repository_directory, 'hosts')
        with open(hosts_file_path, 'w') as hosts_file:
            if self.Parameters.host_group:
                hosts_file.write("[%s]\n" % self.Parameters.host_group)
            hosts_file.write("localhost")

        process.run_process(
            [
                os.path.join(ansible_directory, 'bin', 'python'),
                os.path.join(ansible_directory, 'bin', 'ansible-playbook'),
                self.Parameters.yml_file_path,
                '-vv',
                '-i', hosts_file_path,
                '-c', 'local',
                '-e', 'working_dir="%s" dir="%s"' % (image_sources_directory, image_sources_directory),
                '-e', ' '.join('%s="%s"' % (key, value) for key, value in self.Parameters.parameters.iteritems())
            ],
            log_prefix='ansible',
            work_dir=repository_directory,
            check=False
        )

        return image_sources_directory

    def create_image_sources_resource(self, image_sources_directory):
        resource = resource_types.MARKET_DOCKER_RESOURCE(
            task=self,
            description='description',
            path=image_sources_directory,
        )
        sdk2.ResourceData(resource).ready()
        return resource.id

    def build_docker_image(self, image_sources_resource_id):
        input_parameters = {
            BuildDockerImageV6.PackagedResource.name: image_sources_resource_id,
            BuildDockerImageV6.RegistryTags.name: ['%s:%s' % (self.Parameters.image_name, self.Parameters.image_tag)],
            BuildDockerImageV6.RegistryLogin.name: 'robot-mrkt-ci',
            BuildDockerImageV6.VaultItemName.name: 'MARKET_MULTITESTINGS_DOCKER_REGISTRY_TOKEN',
            BuildDockerImageV6.VaultItemOwner.name: 'MARKETSRE',
        }
        self.Context.children.append(
            sdk2.Task["BUILD_DOCKER_IMAGE_V6"](
                self,
                description='Launch building of {}:{} image from {} task'.format(self.Parameters.image_name, self.Parameters.image_tag, self.id),
                notifications=self.Parameters.notifications,
                **input_parameters
            ).enqueue().id
        )
        raise sdk2.WaitTask(
            self.Context.children,
            list(ctt.Status.Group.FINISH + ctt.Status.Group.BREAK),
            wait_all=True,
            timeout=8 * 60 * 60,  # 8 часов, потому что построение образов может ждать в очереди.
        )
