# coding: utf-8

from __future__ import unicode_literals, division
import logging
from sandbox import sdk2
import sandbox.projects.common.environments as env

import json
import requests
import math


class TicketFields(object):
    def __init__(self, market_project_id):
        # Ключ поля Front
        self.front_field_id = '{}--frontWebEstimation'.format(market_project_id)
        # Ключ поля iOS
        self.ios_field_id = '{}--iosEstimation'.format(market_project_id)
        # Ключ поля Android
        self.android_field_id = '{}--androidEstimation'.format(market_project_id)
        # Ключ поля Checkouter
        self.checkouter_field_id = '{}--checkouterEstimation'.format(market_project_id)
        # Ключ поля Report
        self.report_field_id = '{}--reportEstimation'.format(market_project_id)
        # Ключ поля mCRM
        self.mcrm_field_id = '{}--mcrmEstimation'.format(market_project_id)
        # Ключ поля oCRM
        self.ocrm_field_id = '{}--ocrmEstimation'.format(market_project_id)
        # Ключ поля pers
        self.pers_field_id = '{}--persEstimation'.format(market_project_id)

        # Ключ поля Reach
        self.reach_field_id = 'reach'
        # Ключ поля Impact
        self.impact_field_id = '{}--impact'.format(market_project_id)
        # Ключ поля Confidence
        self.confidence_field_id = '{}--confidence'.format(market_project_id)
        # Ключ поля Effort
        self.effort_field_id = '{}--riceEffort'.format(market_project_id)
        # Ключ поля Коэффициент-k
        self.coefficient_field_id = 'kValue'

        # Ключ поля RICE Score
        self.rice_field_id = '{}--riceScore'.format(market_project_id)
        # Ключ поля RIC Score
        self.ric_field_id = '{}--ricScore'.format(market_project_id)


seconds_in_day = 60 * 60 * 24

impact_values_map = {
    'сильное (ключевая метрика)': 10,
    'среднее (ключевая метрика)': 7,
    'слабое (ключевая метрика)': 5,
    'сильное (неключевая метрика)': 3,
    'среднее (неключевая метрика)': 1,
}

confidence_values_map = {
    'Есть эксперимент': 10,
    'Есть исследование/аналитика': 5,
    'Есть >10 запросов от пользователей': 1,
    'Частное мнение': 0.1,
}


class OfflineExperienceUpdateBoard(sdk2.Task):
    class Requirements(sdk2.Requirements):
        environments = [env.PipEnvironment("isodate"), env.PipEnvironment("datetime")]

    class Parameters(sdk2.Task.Parameters):
        st_queue_filter = sdk2.parameters.Integer("Startrek queue filter id", required=True)
        st_queue_name = sdk2.parameters.String("Startrek queue name", required=True)
        st_secret = sdk2.parameters.String("Startrek secret", required=True)
        token_name = sdk2.parameters.String("Token name in secret", required=True)

    def on_execute(self):
        st_queue_filter = self.Parameters.st_queue_filter
        st_queue_name = self.Parameters.st_queue_name
        token_name = self.Parameters.token_name

        secret = sdk2.yav.Secret(self.Parameters.st_secret)
        oauth_token = secret.data()[token_name]

        filter = {"query": 'Filter: {}'.format(st_queue_filter)}
        headers = {"Authorization": "OAuth {}".format(oauth_token)}
        st_queue_id = self.get_queue_id(st_queue_name, headers)
        fields = TicketFields(st_queue_id)

        logging.info("✨✨✨✨✨✨")
        logging.info(
            'Queue name - {}, queue id - {}, filter id - {}, token_name {}'.format(
                st_queue_name, st_queue_id, st_queue_filter, token_name
            )
        )

        tickets_count = requests.post(
            "https://st-api.yandex-team.ru/v2/issues/_count",
            json=filter,
            headers=headers,
        ).json()

        logging.info("Tikets count {}".format(tickets_count))

        tikets_per_page = 100
        for page_num in [i + 1 for i in range(int(math.ceil(tickets_count / tikets_per_page)))]:
            response = requests.post(
                "https://st-api.yandex-team.ru/v2/issues/_search?page={}&perPage={}".format(page_num, tikets_per_page),
                json=filter,
                headers=headers,
            )

            if response.status_code not in [200, 201]:
                logging.info(
                    "Oh no! 😳 Tickets not acquired, error occured: response: \"{}\"; status code: {}".format(
                        response.text, response.status_code
                    )
                )
                continue

            all_tickets = list(response.json())

            logging.info("Found tickets count in page {} is '{}'".format(page_num, len(all_tickets)))

            for ticket in all_tickets:
                ticket_id = ticket["key"]

                logging.info("💫💫💫💫💫💫💫")
                logging.info("Started calculating Effort for '{}'".format(ticket_id))
                effort = self.calculate_effort(ticket, fields)
                logging.info("✅ Finished calculating Effort for '{}'; Effort is {}".format(ticket_id, effort))

                logging.info("Started calculating RIC Score for '{}'".format(ticket_id))
                ric_score = round(self.calculate_ric_score(ticket, fields), 2)
                logging.info("✅ Finished calculating RIC Score for '{}'; RIC Score is {}".format(ticket_id, ric_score))

                # если больше или равно 99w или effort равен 0, считаем, что не оценено
                is_valid_effort = True if effort / 5 < 99 and effort != 0 else False

                if not is_valid_effort:
                    logging.info(
                        "Effort is not valid (> 99w or 0) for calculating RICE Score. "
                        "Please estimate work duration for all platforms in the ticket '{}'".format(ticket_id)
                    )

                logging.info("Started calculating RICE Score for '{}'".format(ticket_id))
                rice_score = round(ric_score / effort, 2) if is_valid_effort and ric_score else None
                logging.info(
                    "✅ Finished calculating RICE Score for '{}'; RICE Score is {}".format(ticket_id, rice_score)
                )

                field_values = {}

                field_values[fields.effort_field_id] = self.duration_in_iso_8601(effort) if effort else None
                field_values[fields.rice_field_id] = rice_score
                field_values[fields.ric_field_id] = ric_score if ric_score else None

                logging.info(
                    "Setting Effort '{}', RIC Score '{}', RICE Score '{}' for '{}'".format(
                        field_values[fields.effort_field_id],
                        field_values[fields.ric_field_id],
                        field_values[fields.rice_field_id],
                        ticket_id,
                    )
                )
                self.set_field_values(ticket_id, field_values, oauth_token)

    def get_queue_id(self, queue_name, headers):
        response = requests.get(
            "https://st-api.yandex-team.ru/v2/queues/{}/localFields".format(queue_name),
            headers=headers,
        )

        if response.status_code not in [200, 201]:
            logging.info(
                "Oh no! 😳 Failed to get local fields: response: \"{}\"; status code: {}".format(
                    response.text, response.status_code
                )
            )
            raise RuntimeError("Failed to get local fields")

        for field in list(response.json()):
            return field["id"].split('--')[0]

        raise RuntimeError("There is no local fields in queue {}".format(queue_name))

    def calculate_effort(self, ticket, fields):
        front_estimation = self.duration_in_work_days(ticket.get(fields.front_field_id, None))
        ios_estimation = self.duration_in_work_days(ticket.get(fields.ios_field_id, None))
        android_estimation = self.duration_in_work_days(ticket.get(fields.android_field_id, None))
        checkouter_estimation = self.duration_in_work_days(ticket.get(fields.checkouter_field_id, None))
        report_estimation = self.duration_in_work_days(ticket.get(fields.report_field_id, None))
        mcrm_estimation = self.duration_in_work_days(ticket.get(fields.mcrm_field_id, None))
        ocrm_estimation = self.duration_in_work_days(ticket.get(fields.ocrm_field_id, None))
        pers_estimation = self.duration_in_work_days(ticket.get(fields.pers_field_id, None))

        return sum(
            [
                front_estimation,
                ios_estimation,
                android_estimation,
                checkouter_estimation,
                report_estimation,
                mcrm_estimation,
                ocrm_estimation,
                pers_estimation,
            ]
        )

    def calculate_ric_score(self, ticket, fields):
        reach = ticket.get(fields.reach_field_id, 0)
        impact = impact_values_map.get(ticket.get(fields.impact_field_id, None), 0)
        confidence = confidence_values_map.get(ticket.get(fields.confidence_field_id, None), 0)
        coefficient = ticket.get(fields.coefficient_field_id, 1)
        return reach * impact * confidence * coefficient

    def duration_in_work_days(self, duration_in_iso_8601):
        from isodate import parse_duration

        if duration_in_iso_8601:
            time_delta = parse_duration(duration_in_iso_8601)
            calendar_days = time_delta.days
            weeks = int(calendar_days / 7)
            # если больше или равно 99w, считаем, что не оценено
            if weeks >= 99:
                return 0
            else:
                remainder_days = calendar_days % 7
                work_days = weeks * 5 + remainder_days
                seconds = time_delta.seconds
                # нам не нужно целое число, поэтому используем /, а не //
                duration = work_days + (seconds / seconds_in_day)
                return duration
        else:
            return 0

    def duration_in_iso_8601(self, duration_in_work_days):
        import datetime
        from isodate import duration_isoformat

        if duration_in_work_days:
            time_delta = datetime.timedelta(days=duration_in_work_days)
            duration = duration_isoformat(time_delta)
            return duration
        else:
            return ''

    def set_field_values(self, ticket_id, field_values, oauth_token):
        request = requests.patch(
            "https://st-api.yandex-team.ru/v2/issues/{}".format(ticket_id),
            json.dumps(field_values, indent=4),
            headers={"Authorization": "OAuth {}".format(oauth_token)},
        )

        try:
            request.raise_for_status()
            logging.info("🎉🎉🎉 Hooray! Finished job for '{}'".format(ticket_id))
        except requests.HTTPError:
            error_msg = "Oops! 🥺 ST HTTP error: response: \"{}\"; status code: {}".format(
                request.text, request.status_code
            )
            logging.info(error_msg)
            raise
