# coding=UTF8
import requests
import logging
import json
import datetime
from sandbox.projects.release_machine.helpers.staff_helper import StaffApi


class Escalator:
    def __init__(self,
                 yav_tokens,
                 escalation_chat_id,
                 checkouter_filter_id,
                 ios_filter_id,
                 android_filter_id,
                 web_filter_id,
                 qa_filter_id):
        self.startrek_oauth_token = yav_tokens.data()['startrek_token']
        self.telegram_bot_token = yav_tokens.data()['telegram_bot_token']
        self.abc_token = yav_tokens.data()['abc_token']
        self.telegram_api_url = 'https://api.telegram.org'
        self._staff = StaffApi(self.startrek_oauth_token)
        self.staff_url = "https://staff.yandex-team.ru/{}"
        self.logger = logging.getLogger(__name__)
        self.abc_api_url = 'https://abc-back.yandex-team.ru/api/v4/duty/shifts/?service__slug={}&date_from={}&date_to={}&with_watcher=1'
        self.startrek_api_url = 'https://st-api.yandex-team.ru/v2'
        self.slug_id='offline_ux'
        self.startrek_issues_url = 'https://st.yandex-team.ru/issues/'
        self.checkouter_filter_id=checkouter_filter_id
        self.ios_filter_id=ios_filter_id
        self.android_filter_id=android_filter_id
        self.web_filter_id=web_filter_id
        self.qa_filter_id=qa_filter_id
        self.escalation_chat_id=escalation_chat_id

    def do_escalation(self):
        today = datetime.datetime.today().replace(hour=0, minute=0, second=0, microsecond=0)
        duties=self.get_current_duties(today)
        android_duty_person=self.get_duty('Android', duties)
        ios_duty_person=self.get_duty('iOS', duties)
        checkouter_duty_person=self.get_duty('Checkouter', duties)
        web_duty_person=self.get_duty('Web',duties)
        qa_duty_person=self.get_duty('QA', duties)
        message='<b>Необработанные обращения в поддержке контура:</b>'

        android_issues=self.get_issues(self.android_filter_id)
        issues_count=0
        if android_issues:
            message=self.add_issues_to_message(message, android_issues, android_duty_person,
                                               'Android', self.android_filter_id)
            issues_count+=len(android_issues)

        ios_issues=self.get_issues(self.ios_filter_id)
        if ios_issues:
            message=self.add_issues_to_message(message, ios_issues, ios_duty_person,
                                               'iOS', self.ios_filter_id)
            issues_count+=len(ios_issues)

        checkouter_issues=self.get_issues(self.checkouter_filter_id)
        if checkouter_issues:
            message=self.add_issues_to_message(message, checkouter_issues, checkouter_duty_person,
                                               'Checkouter', self.checkouter_filter_id)
            issues_count+=len(checkouter_issues)

        web_issues=self.get_issues(self.web_filter_id)
        if web_issues:
            message=self.add_issues_to_message(message, web_issues, web_duty_person,
                                               'Web', self.web_filter_id)
            issues_count+=len(web_issues)

        qa_issues=self.get_issues(self.qa_filter_id)
        if qa_issues:
            message=self.add_issues_to_message(message, qa_issues, qa_duty_person,
                                               'QA', self.qa_filter_id)
            issues_count+=len(qa_issues)

        if issues_count>0:
            self.logger.info('issues count: {}'.format(issues_count))
            message=message +'\n\n Обработайте тикеты и проставьте тег \'Обработано\''
            self.send_telegram_message(message)





    def get_current_duties(self, today):
        duties = requests.get(
                self.abc_api_url.format(self.slug_id, today.strftime("%Y-%m-%d"), today.strftime("%Y-%m-%d")),
                headers={'Authorization': 'OAuth {}'.format(self.abc_token)}).json()['results']
        self.logger.info("Duties found: {}".format(duties))
        return duties

    def get_issues(self, filter_id):
        response_json= requests.get(
            '{}/issues?query=Filter:{}'.format(self.startrek_api_url, filter_id),
            headers=self.startrek_headers()
        ).json()
        self.logger.info('found {} issues for filter {}'.format(len(response_json), filter_id))
        return response_json



    def add_issues_to_message(self, message, issues, duty_person, duty_type, filter_id):
        issue_text_pattern_with_assignee = "<a href=\"https://st.yandex-team.ru/{}\">    {}</a>, ответственный @{}"
        self.logger.info("adding issues: {} for {}".format(issues,duty_person))

        if len(issues) > 0:
            message=message+'\n\n<i>{}</i>:'.format(duty_type)
        for issue in issues:
            if "assignee" in issue:
                message = message \
                        + "\n" \
                        + issue_text_pattern_with_assignee.format(issue["key"],  issue["key"],
                                                                self.get_telegram_contact(issue["assignee"]['id']))
            else:
                message = message \
                        + "\n" \
                        + issue_text_pattern_with_assignee.format(issue["key"],  issue["key"],
                                                                  self.get_telegram_contact(duty_person))
        if len(issues) > 1:
            message = message + "\n" + \
                      ("<a href=\"{}{}\">Список в трекере</a>".format(self.startrek_issues_url, filter_id))

        return message

    def send_telegram_message(self, message):
        data = json.dumps({
            "chat_id": self.escalation_chat_id,
            "text": message,
            "parse_mode": "HTML",
            "method": "sendMessage"
        })
        result = requests.post(
            '{}/bot{}/sendMessage'.format(self.telegram_api_url, self.telegram_bot_token),
            headers={'Content-Type': 'application/json'},
            data=data
        ).json()
        self.logger.info("Sending notification result {}".format(result))
        return result

    def get_telegram_contact(self, login):
        contacts = self._staff.get_person_profile_property(login, 'telegram_accounts')['telegram_accounts']
        for contact in contacts:
            if 'bot' not in contact['value']:
                return contact['value']

    def get_duty(self, duty_type, duties):
        return {
            'Android' : [x['person']['login'] for x in duties if x['schedule']['name'] == 'Android'],
            'iOS' : [x['person']['login'] for x in duties if x['schedule']['name'] == 'iOS'],
            'Checkouter' : [x['person']['login'] for x in duties if x['schedule']['name'] == 'Checkouter'],
            'QA' : [x['person']['login'] for x in duties if x['schedule']['name'] == 'QA'],
            'Web' : [x['person']['login'] for x in duties if x['schedule']['name'] == 'Web']
        }[duty_type]

    def startrek_headers(self):
        return {'Authorization': 'Bearer {}'.format(self.startrek_oauth_token), 'Content-Type': 'application/json'}
